<?php

namespace App\Controller;

use App\Entity\Main\DashboardUserExport;
use App\Service\EmailsTemplateSender;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\Intl\Countries;
use Symfony\Component\Routing\Annotation\Route;

\Locale::setDefault('en');

/**
 * @Route("/dashboard")
 */
class DashboardController extends AbstractController
{
    protected $em;
    private $emailsTemplateSender;
    private $params;

    public function __construct(EntityManagerInterface $entityManager,
                                EmailsTemplateSender   $emailsTemplateSender,
                                ParameterBagInterface  $params)
    {
        $this->em = $entityManager;
        $this->emailsTemplateSender = $emailsTemplateSender;
        $this->params = $params;
    }

    /**
     * @Route("/", name="app_dashboard")
     */
    public function index(): Response
    {
        return $this->render('dashboard/index.html.twig', [
            'error' => '',
            'listSites' => $this->listSites(),
            'categories' => $this->listSitesCategory(),
            'countries' => Countries::getNames()
        ]);
    }

    /**
     * @Route("/map", name="app_dashboard_mappage")
     */
    public function mapPage(): Response
    {
        return $this->render('dashboard/map.html.twig');
    }

    /**
     * @Route("/map_24_25", name="app_dashboard_map_24_25")
     */
    public function mapSeason24_25(): Response
    {
        return $this->render('dashboard/map24_25.html.twig');
    }
    // get all list sites from database in alphabetical order (order by zone ASC)
    public function listSites(): array
    {
        //  connect to the database
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);
        // requet sql to select all site from table site order by zone
        $sqlListSites = "SELECT zone FROM site ORDER BY CASE 
                              WHEN order_zone IS NULL THEN 1 
                            ELSE 0 
                            END,
                      order_zone ASC ;";
        $sites = $connDB->fetchAll($sqlListSites);
        // return list site in an array
        return $sites;
    }

    // get all list sites from database in alphabetical order (order by zone ASC)
    public function listSitesCategory(): array
    {
        //  connect to the database
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);
        // requet sql to select all site from table site order by category
        $sqlListSites = "SELECT DISTINCT(category), category_en as categoryEn FROM site ORDER BY categoryEn ASC";
        $categories = $connDB->fetchAll($sqlListSites);
        // return list site categories in an array
        return $categories;
    }

    // get list of sites at least they have row in table data
    public function getValidSitesBySeason($season): array
    {
        // connect to the database
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL"), 'charset' => 'utf8']);
        // requet sql select distinct sites in alphabetical order and at least they have row in table data
        $sqlListSites = 'SELECT DISTINCT(s.zone) 
                         FROM data_dashboard d 
                         LEFT JOIN site s on d.site_id = s.id 
                         WHERE d.season ="' . $season . '" 
                         ORDER BY CASE 
                              WHEN s.order_zone IS NULL THEN 1 
                            ELSE 0 
                            END,
                            s.order_zone ASC ;';
        $sites = $connDB->fetchAll($sqlListSites);
        $listSites = [];
        foreach ($sites as $site) {
            array_push($listSites, $site["zone"]);
        }
        return $listSites;
    }


    // start code View 2

    /**
     * @Route("/init-data-view2", name="init_data_view2_ajax")
     */
    public function initDataView2(Request $request): JsonResponse
    {
        $data = $request->get('data');
        $result = $this->getDataView2($data['season'], $data['virus'],  $data['virusColor']);
        return new JsonResponse([
            'status' => 200,
            'data' => $result['data'],
            'maxPositivityRate' => $result['maxPositivityRate'],
            'totalVirus' => $result['totalVirus']
        ]);
    }

    /**
     * @Route("/statistics-view2", name="statistics_view2_ajax")
     */
    public function dataStatisticsView2(Request $request): JsonResponse
    {
        // get data of sites, weeks and seasons send it from the form
        $data = $request->get('data');
        $result = $this->getDataView2($data["season"], $data["virus"], $data['virusColor']);

        return new JsonResponse([
            'status' => 200,
            'data' => $result['data'],
            'maxPositivityRate' => $result['maxPositivityRate'],
            'totalVirus' => $result['totalVirus']
        ]);

    }

    public function getDataView2($season, $virus = null, $virusColor): array
    {
        $result = $response = array();
        switch ($virus) {
            case  'both_positive_result' :
                $result = $this->querySQLView2('both_positive_result', null, $season);
                break;
            case  'both_lab2_sarscov2_yesno':
                $result = $this->querySQLView2('both_lab2_sarscov2_yesno', 'both_lab2_sarscov2', $season);
                break;
            case  'both_lab2_rs_yesno':
                $result = $this->querySQLView2('both_lab2_rs_yesno', 'both_lab2_rs', $season);
                break;
            case  'orv_plus_sans_rsv_sans_sars_cov_2':
                $result = $this->querySQLView2('orv_plus_sans_rsv_sans_sars_cov_2', 'test_orv_sans_rsv_sans_sars_cov_2', $season);
                break;
            default:
                $result = $this->querySQLView2('both_positive_result', null, $season);
        }

        $max = 0;
        $totalVirus = 0;
        foreach ($result as &$item) {
            if ($item['percent'] > $max) {
                $max = $item['percent'] + 5;
            }
            if ($item['cases'] == null || $item['cases'] == 0) {
                $item['cases'] = "NA";
            } else {
                $totalVirus += $item['cases'];
            }

            if ($item['percent'] == null || $item['percent'] == 0) {
                $item['color'] = "#808080";
                $item['percent'] = "NA";
            } else {
                $startColor = $this->determineStartColor($virusColor);
                $item['color'] = $this->interpolateColor($item['percent'], 0, 100, $startColor, $virusColor);
                $item['percent'] = $item['percent'] . "%";

            }
        }
        unset($item);
        $response['data'] = $result;
        $response['maxPositivityRate'] = $max;
        $response['totalVirus'] = $totalVirus;
        return $response;
    }

    public function querySQLView2($virus1, $virus2, $season): array
    {
        /* connexion à la base de données */
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $sqlQuery = 'SELECT 
                     s.id AS site_id,
                     s.zone AS zone,
                     s.latitude AS latitude,
                     s.longitude AS longitude,
                     s.site_name_season_22 AS site_name_season_22,
                     s.site_name_season_23 AS site_name_season_23,';
        if ($virus1 == "both_positive_result") {
            $sqlQuery .= '(SELECT COUNT(*) FROM data_dashboard AS d2 WHERE d2.site_id = s.id AND d2.season = "' . $season . '" AND (d2.' . $virus1 . ' = 1 OR d2.' . $virus1 . ' = "1" )';
            $sqlQuery .= ') AS cases, (ROUND((SELECT COUNT(*) FROM data_dashboard AS d2 WHERE d2.site_id = s.id AND d2.season =  "' . $season . '"  AND (d2.' . $virus1 . ' = 1 OR d2.' . $virus1 . ' = "1")';
            $sqlQuery .= ') * 100.0 / (SELECT COUNT(*) FROM data_dashboard AS d2  WHERE d2.site_id = s.id AND d2.season =  "' . $season . '"  AND d2.site_id NOT IN ("2","13") ';
            $sqlQuery .= '),0)) AS percent';
        }else {
            $sqlQuery .= '(SELECT COUNT(*) FROM data_dashboard AS d2 WHERE d2.site_id = s.id AND d2.season = "' . $season . '" AND (d2.' . $virus1 . ' = 1 OR d2.' . $virus1 . ' = "1" )';
            $sqlQuery .= ') AS cases,';
            $sqlQuery .= '(ROUND((SELECT COUNT(*) FROM data_dashboard AS d2 WHERE d2.site_id = s.id AND d2.season =  "' . $season . '"  AND (d2.' . $virus1 . ' = 1 OR d2.' . $virus1 . ' = "1" ) ';
            $sqlQuery .= ') * 100.0 /';
            $sqlQuery .= '(SELECT COUNT(*) FROM data_dashboard AS d2  WHERE d2.site_id = s.id AND d2.season =  "' . $season . '"  AND (d2.' . $virus2 . ' = 1 OR d2.' . $virus2 . ' = "1" ) ';
            $sqlQuery .= '),0)) AS percent';
        }

        $sqlQuery .= ' FROM data_dashboard AS d
                    JOIN site AS s
                    ON d.site_id = s.id
                    WHERE d.season =  "' . $season . '" GROUP BY zone;';
        return $connDB->fetchAll($sqlQuery);
    }

    public function interpolateColor($value, $min, $max, $startColor, $endColor)
    {
        list($r1, $g1, $b1) = sscanf($startColor, "#%02x%02x%02x");
        list($r2, $g2, $b2) = sscanf($endColor, "#%02x%02x%02x");

        // Calculer le pourcentage
        $ratio = ($value - $min) / ($max - $min);

        // Interpoler chaque composante
        $r = round($r1 + $ratio * ($r2 - $r1));
        $g = round($g1 + $ratio * ($g2 - $g1));
        $b = round($b1 + $ratio * ($b2 - $b1));

        // Retourner la couleur interpolée en format hexadécimal
        return sprintf("#%02x%02x%02x", $r, $g, $b);
    }

    private function determineStartColor($virusColor): string
    {
        //based of the virus color to determine the start color
        $startCodeColor = "#CEB0C6FF";
        if ($virusColor === "#D9144C") {
            $startCodeColor = "#EE97B0FF";
        } elseif ($virusColor === "#6E9890") {
            $startCodeColor = "#B4CAC6FF";
        } elseif ($virusColor === "#D9D632") {
            $startCodeColor = "#F2F1BCFF";
        }
        return $startCodeColor;
    }

    // END code View 2


    // start code View 3

    /**
     * @Route("/init-data-view3", name="init_data_view3_ajax")
     */
    public function initDataView3(): JsonResponse
    {
        $season = ["21_22", "22_23"];
        $weeksSeason21_22 = $this->getAllNumberWeeksSeason("2021-09-01", "2022-10-31");
        $weeksSeason22_23 = $this->getAllNumberWeeksSeason("2022-11-01", "2023-10-31");
        $sites = [];
        $site21 = $this->getValidSitesBySeason("21_22");
        $site22 = $this->getValidSitesBySeason("22_23");

        $weekNumbersKeys = [];
        foreach ($weeksSeason21_22 as $week) {
                $weekNumbersKeys[] = $week;
        }
        foreach ($weeksSeason22_23 as $week) {
            $weekNumbersKeys[] = $week;
        }

        foreach ($site21 as $site) {
            if (!in_array($site, $sites)) {
                $sites[] = $site;
            }
        }
        foreach ($site22 as $site) {
            if (!in_array($site, $sites)) {
                $sites[] = $site;
            }
        }
        // get data  view 3
        $dataView3 = $this->getDataView3($season, null, null, null, $weekNumbersKeys);

        return new JsonResponse([
            'status' => 200,
            'listNumberWeeks' => $weekNumbersKeys,
            'bar1' => $dataView3['bar1'],
            'lineData' => $dataView3['lineData'],
            'sites' => $sites
        ]);
    }

    /**
     * @Route("/statistics-view3", name="statistics_view3_ajax")
     */
    public function dataStatisticsView3(Request $request): JsonResponse
    {
        // get data of sites, weeks and seasons send it from the form
        $data = $request->get('data');
        $season = ["21_22", "22_23"];
        $site = $data["site"];
        $zone = $data["zone"];
        $virus = $data["virus"];

        $weeksSeason21_22 = $this->getAllNumberWeeksSeason("2021-09-01", "2022-10-31");
        $weeksSeason22_23 = $this->getAllNumberWeeksSeason("2022-11-01", "2023-10-31");

        $weekNumbersKeys = [];
        foreach ($weeksSeason21_22 as $week) {
                $weekNumbersKeys[] = $week;
        }
        foreach ($weeksSeason22_23 as $week) {
            $weekNumbersKeys[] = $week;
        }
        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataView3($season, $virus, $zone, $site, $weekNumbersKeys)]);

    }

    public function getDataView3($season, $virus = null, $zone = null, $site, $weeks): array
    {

        // init array result
        $lineData = array_combine(array_keys($weeks), array_fill(0, count($weeks), [0, 0]));
        $bar1Data = array_combine(array_keys($weeks), array_fill(0, count($weeks), 0));

        $lineEnrolled = $bar1 = array();
        $sites = [];
        switch ($virus) {
            case  'both_positive_result' :
                $lineEnrolled = $this->querySQLView3($zone, $site, $virus);
//                Influenza – all patients (NA for Canada and St Petersburg)
                $bar1 = $this->querySQLView3($zone, $site, "influenza");
                break;
            case  'both_lab2_sarscov2_yesno':
                $lineEnrolled = $this->querySQLView3($zone, $site, $virus);
                $bar1 = $this->querySQLView3($zone, $site, "both_lab2_sarscov2");
                break;
            case  'both_lab2_rs_yesno':
                $lineEnrolled = $this->querySQLView3($zone, $site, $virus);
                $bar1 = $this->querySQLView3($zone, $site, "both_lab2_rs");
                break;
            case  'orv_plus_sans_rsv_sans_sars_cov_2':
                $lineEnrolled = $this->querySQLView3($zone, $site, $virus);
                $bar1 = $this->querySQLView3($zone, $site, "test_orv_sans_rsv_sans_sars_cov_2");
                break;
            default:
                $lineEnrolled = $this->querySQLView3($zone, $site, "orv_plus");
                $bar1 = $this->querySQLView3($zone, $site, "all");
                break;
        }


        foreach ($lineEnrolled as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $lineData[$key] = [$key, $item["total"] ? $lineData[$key][1] + (int)$item["total"] : $lineData[$key][1] + 0];
            if (!in_array($item["zone"], $sites)) {
                $sites[] = $item["zone"];
            }
        }

        foreach ($bar1 as $item) {
            $weekNumber = $item["week_number"];

            $key = array_search($weekNumber, $weeks);
            $bar1Data[$key] += (int)$item["total"];
            if (!in_array($item["zone"], $sites)) {
                $sites[] = $item["zone"];
            }

        }

        return array(
            'sites' => $sites,
            'listNumberWeeks' => $weeks,
            'lineData' => $lineData,
            'bar1' => $bar1Data
        );
    }

    public function querySQLView3($zone = null, $site = null, $variable): array
    {
        /* connexion à la base de données */
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $sqlQuery = 'SELECT COUNT(*) AS total,
                            d.week_number,
                            s.zone
                     FROM data_dashboard d
                     LEFT JOIN site s ON d.site_id = s.id
                     WHERE d.week_number is not null and d.site_id is not null';

        if ($variable == "orv_plus") {
            $sqlQuery .= '  AND ((d.' . $variable . ' = 1 OR d.' . $variable . ' = "1") OR (d.both_positive_result = 1 OR d.both_positive_result = "1")) ';
        }
        if ($variable == "influenza") {
            $sqlQuery .= ' AND s.id NOT IN (2,13) ';
        }
        if (($variable != "all") && ($variable != "influenza") && ($variable != "orv_plus")) {
            $sqlQuery .= ' AND (d.' . $variable . ' = 1 OR d.' . $variable . ' = "1") ';
        }

        // filter condution
        if (!empty($site) && $site !== "all") {
            $sqlQuery .= ' AND s.zone IN("' . $site . '")';
        }

        if (!empty($zone) && $zone !== "all") {
            $sqlQuery .= ' AND s.category_en IN("' . $zone . '")';
        }

        $sqlQuery .= '  GROUP BY d.week_number, d.site_id';
        return $connDB->fetchAll($sqlQuery);
    }


    // END code View 3

    // start code View 4 Distribution of patients by age group

    /**
     * @Route("/init-data-view4", name="init_data_view4_ajax")
     */
    public function initDataView4(): JsonResponse
    {
        $sites = [];
        $site21 = $this->getValidSitesBySeason("21_22");
        $site22 = $this->getValidSitesBySeason("22_23");

        $weeksSeason21_22 = $this->getAllNumberWeeksSeason("2021-09-01", "2022-10-31");
        $weeksSeason22_23 = $this->getAllNumberWeeksSeason("2022-11-01", "2023-10-31");
        $weekNumbersKeys = [];
        foreach ($weeksSeason21_22 as $week) {
                $weekNumbersKeys[] = $week;
        }
        foreach ($weeksSeason22_23 as $week) {
                $weekNumbersKeys[] = $week;
        }

        foreach ($site21 as $site) {
            if (!in_array($site, $sites)) {
                $sites[] = $site;
            }
        }
        foreach ($site22 as $site) {
            if (!in_array($site, $sites)) {
                $sites[] = $site;
            }
        }
        // get data view 4
        $dataView4 = $this->getDataView4($sites, null, $weekNumbersKeys, true);

        return new JsonResponse([
            'status' => 200,
            'listNumberWeeks' => $weekNumbersKeys,
            'bar1' => $dataView4['bar1'],
            'bar2' => $dataView4['bar2'],
            'bar3' => $dataView4['bar3'],
            'bar4' => $dataView4['bar4'],
            'bar5' => $dataView4['bar5'],
            'bar6' => $dataView4['bar6'],
            'lineData' => $dataView4['lineData'],
            'sites' => $dataView4['sites'],
            'totalIncluded' => $dataView4['totalIncluded'],
        ]);
    }

    public function getAllNumberWeeksSeason($startSeason, $endSeason): array
    {
        // Set the start date to November 1, date of start season
        $startDate = new \DateTime($startSeason);
        // Set the end date to the current date
        $endDate = new \DateTime($endSeason);
        // Create an empty array to store the week numbers
        $weekNumbers = [];
        // Iterate over each week in the date range and add the week number to the array
        while ($startDate <= $endDate) {
            // Get the week number and concat with --Year(YYYY)
            $weekNumbers[] = $startDate->format('W') . '--' . $startDate->format('Y');
            // add a week to the date
            $startDate->modify('+1 week');
        }
        return $weekNumbers;
    }

    public function getlistWeeksNumberBySeason($season): array
    {
        // connect to the database
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);
        // requet sql select distinct sites in alphabetical order and at least they have row in table data
        $sqlListWeeksNumber = 'SELECT DISTINCT(week_number) AS week_number FROM data_dashboard  WHERE season IN ("' . implode("\",\"", $season) . '") ';
        $result = $connDB->fetchAll($sqlListWeeksNumber);
        $listWeeksNumber = [];
        foreach ($result as $week) {
            array_push($listWeeksNumber, $week["week_number"]);
        }
        // returns an array containing the list of Week Number valid
        return $listWeeksNumber;
    }

    /**
     * @Route("/statistics-view4", name="statistics_view4_ajax")
     */
    public function dataStatisticsView4(Request $request): JsonResponse
    {
        // get data of sites, weeks and seasons send it from the form
        $data = $request->get('data');
        $site = $data["site"];
        $zone = $data["zone"];
        $weeksSeason21_22 = $this->getAllNumberWeeksSeason("2021-09-01", "2022-10-31");
        $weeksSeason22_23 = $this->getAllNumberWeeksSeason("2022-11-01", "2023-10-31");
        $weekNumbersKeys = [];
        foreach ($weeksSeason21_22 as $week) {
            $weekNumbersKeys[] = $week;
        }
        foreach ($weeksSeason22_23 as $week) {
            $weekNumbersKeys[] = $week;
        }
        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataView4($site, $zone, $weekNumbersKeys, false)]);

    }

    public function getDataView4($sites, $zone = null, $weeks, $firstCall): array
    {

        // init array result
        $lineData = array_combine(array_keys($weeks), array_fill(0, count($weeks), [0, 0]));
        $bar1Data = $bar2Data = $bar3Data = $bar4Data = $bar5Data = $bar6Data = array_combine(array_keys($weeks), array_fill(0, count($weeks), 0));
        $listSites = [];
        //data enrolled
        $lineEnrolled = $this->querySQLView4($sites, $zone, 'included', $firstCall);
        $bar1 = $this->querySQLView4($sites, $zone, 'both_lab_flu_h1n1', $firstCall);
        $bar2 = $this->querySQLView4($sites, $zone, 'both_lab_flu_h3n2', $firstCall);
        $bar3 = $this->querySQLView4($sites, $zone, 'both_lab_flu_yamagata', $firstCall);
        $bar4 = $this->querySQLView4($sites, $zone, 'both_lab_flu_victoria', $firstCall);
        $bar5 = $this->querySQLView4($sites, $zone, 'both_lab_flu_a_nosub', $firstCall);
        $bar6 = $this->querySQLView4($sites, $zone, 'both_lab_flu_b_nosub', $firstCall);

        $totalIncluded = 0;
        foreach ($lineEnrolled as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $lineData[$key] = [$key, $lineData[$key][1] + (int)$item["included"]];
            $totalIncluded += (int)$item["included"];
            if (!in_array($item["zone"], $listSites)) {
                $listSites[] = $item["zone"];
            }
        }

        foreach ($bar1 as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $bar1Data[$key] += (int)$item["both_lab_flu_h1n1"];
            if (!in_array($item["zone"], $listSites)) {
                $listSites[] = $item["zone"];
            }
        }

        foreach ($bar2 as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $bar2Data[$key] += (int)$item["both_lab_flu_h3n2"];
            if (!in_array($item["zone"], $listSites)) {
                $listSites[] = $item["zone"];
            }
        }

        foreach ($bar3 as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $bar3Data[$key] += (int)$item["both_lab_flu_yamagata"];
            if (!in_array($item["zone"], $listSites)) {
                $listSites[] = $item["zone"];
            }
        }

        foreach ($bar4 as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $bar4Data[$key] += (int)$item["both_lab_flu_victoria"];
            if (!in_array($item["zone"], $listSites)) {
                $listSites[] = $item["zone"];
            }
        }

        foreach ($bar5 as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $bar5Data[$key] += (int)$item["both_lab_flu_a_nosub"];
            if (!in_array($item["zone"], $listSites)) {
                $listSites[] = $item["zone"];
            }
        }

        foreach ($bar6 as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $bar6Data[$key] += (int)$item["both_lab_flu_b_nosub"];
            if (!in_array($item["zone"], $listSites)) {
                $listSites[] = $item["zone"];
            }
        }

        return array(
            'sites' => $listSites,
            'listNumberWeeks' => $weeks,
            'lineData' => $lineData,
            'bar1' => $bar1Data,
            'bar2' => $bar2Data,
            'bar3' => $bar3Data,
            'bar4' => $bar4Data,
            'bar5' => $bar5Data,
            'bar6' => $bar6Data,
            'totalIncluded' => $totalIncluded
        );
    }

    public function querySQLView4($site = null, $category = null, $variable, $firstCall): array
    {
        /* connexion à la base de données */
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $sqlQuery = 'SELECT COUNT(' . $variable . ') AS ' . $variable . ', d.week_number, s.zone
                     FROM data_dashboard d
                     LEFT JOIN site s ON d.site_id = s.id
                     WHERE d.week_number IS NOT NULL AND d.site_id IS NOT NULL AND (d.' . $variable . ' = 1  OR  d.' . $variable . ' = "1") ';

        if ($firstCall) {
            $sqlQuery .= ' AND s.zone IN ("' . implode("\",\"", $site) . '")';
        } else {
            if (!empty($site) && $site !== "all") {
                $sqlQuery .= ' AND s.zone IN("' . $site . '")';
            }
            if (!empty($category) && $category !== "all") {
                $sqlQuery .= ' AND s.category IN("' . $category . '")';
            }
        }

        $sqlQuery .= '   GROUP BY d.week_number, d.site_id;';
        return $connDB->fetchAll($sqlQuery);


    }

    // END code View 4

    // start code View 5 Distribution of patients by age group

    /**
     * @Route("/init-data-view5", name="init_data_view5_ajax")
     */
    public function initDataView5(Request $request): JsonResponse
    {
        $data = $request->get('data');
        $season = $data['season'];
        // get list of sites valid in the selected season ( at least they have row in table data)
        $listSitesHasDataInSelectedSeason = $this->getValidSitesBySeason($season);
        // get data View5
        $dataView5 = $this->getDataView5($season, null, $listSitesHasDataInSelectedSeason);

        return new JsonResponse([
            'status' => 200,
            'current_season' => $season,
            'totalDataBySite' => $dataView5['totalDataBySite'],
            'totalDataAllSites' => $dataView5['totalDataAllSites'],
            'ageLessThan5' => $dataView5['ageLessThan5'],
            'ageBetween5And49' => $dataView5['ageBetween5And49'],
            'ageBetween50And64' => $dataView5['ageBetween50And64'],
            'ageMoreThan65' => $dataView5['ageMoreThan65'],
            'sitesChart' => $dataView5['sitesChart'],
        ]);
    }

    /**
     * @Route("/statistics-view5", name="statistics_view5_ajax")
     */
    public function dataStatisticsView5(Request $request): JsonResponse
    {
        $data = $request->get('data');
        $season = $data['season'];
        $site = $this->getValidSitesBySeason($season);
        $virus = $data['virus'];

        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataView5($season, $virus, $site)]);
    }

    public function getDataView5($season, $virus, $site)
    {
        //  LIST OF SITES BY KEYS
        $sitesKeys = $this->listKeysSites($site);

        // intialize the result in array format and all data by site to 0
        $totalDataBySite = $ageLessThan5Data = $ageBetween5And49Data =
        $ageBetween50And64Data = $ageMoreThan65Data = array_combine(array_keys($sitesKeys), array_fill(0, count($sitesKeys), 0));
        $totalDataAllSites = 0;
        if ($season) {
            $ageLessThan5 = $this->querySQLView5('< 5y', $season, $virus, $site);
            $ageBetween5And49 = $this->querySQLView5('5-49y', $season, $virus, $site);
            $ageBetween50And64 = $this->querySQLView5('50-64y', $season, $virus, $site);
            $ageMoreThan65 = $this->querySQLView5('+65y', $season, $virus, $site);

            $ageGroups = [
                [$ageLessThan5, &$ageLessThan5Data],
                [$ageBetween5And49, &$ageBetween5And49Data],
                [$ageBetween50And64, &$ageBetween50And64Data],
                [$ageMoreThan65, &$ageMoreThan65Data]
            ];

            foreach ($ageGroups as [$ageGroup, &$data]) {
                foreach ($ageGroup as $item) {
                    $zone = $item["zone"];
                    $key = array_search($zone, $sitesKeys);
                    $data[$key] = $item["tranche_age_def3"];
                }
            }

            $sites = [];
            foreach ($site as $item) {
                $key = array_search($item, $sitesKeys);
                $totalDataBySite[$key] = $ageLessThan5Data[$key] + $ageBetween5And49Data[$key] + $ageBetween50And64Data[$key] + $ageMoreThan65Data[$key];
                if ($totalDataBySite[$key] != 0) {
                    $sites[] = $item;
                } elseif ($totalDataBySite[$key] == 0) {
                    unset($totalDataBySite[$key]);
                    unset($ageLessThan5Data[$key]);
                    unset($ageBetween5And49Data[$key]);
                    unset($ageBetween50And64Data[$key]);
                    unset($ageMoreThan65Data[$key]);
                }
            }

            // Aggregate sums for the "All Sites" entry
            $totalAgeLessThan5 = array_sum($ageLessThan5Data);
            $totalAgeBetween5And49 = array_sum($ageBetween5And49Data);
            $totalAgeBetween50And64 = array_sum($ageBetween50And64Data);
            $totalAgeMoreThan65 = array_sum($ageMoreThan65Data);
            $totalAllSites = $totalAgeLessThan5 + $totalAgeBetween5And49 + $totalAgeBetween50And64 + $totalAgeMoreThan65;

            // Add the "All Sites" entry
            $sites[] = "All Sites";
            $totalDataBySite[] = $totalAllSites;
            $ageLessThan5Data[] = $totalAllSites > 0 ? intval(round(($totalAgeLessThan5 / $totalAllSites) * 100)) : 0;
            $ageBetween5And49Data[] = $totalAllSites > 0 ? intval(round(($totalAgeBetween5And49 / $totalAllSites) * 100)) : 0;
            $ageBetween50And64Data[] = $totalAllSites > 0 ? intval(round(($totalAgeBetween50And64 / $totalAllSites) * 100)) : 0;
            $ageMoreThan65Data[] = $totalAllSites > 0 ? intval(round(($totalAgeMoreThan65 / $totalAllSites) * 100)) : 0;


            // rebuild result to match new indexs and keys
            $totalDataBySite = array_values($totalDataBySite);
            $ageLessThan5Data = array_values($ageLessThan5Data);
            $ageBetween5And49Data = array_values($ageBetween5And49Data);
            $ageBetween50And64Data = array_values($ageBetween50And64Data);
            $ageMoreThan65Data = array_values($ageMoreThan65Data);

            //get new keys of new list of sites
            $listKeys = $this->listKeysSites($sites);

            foreach ($sites as $item) {
                $key = array_search($item, $listKeys);
                if ($item !== "All Sites") {
                    $totalDataBySite[$key] = $ageLessThan5Data[$key] + $ageBetween5And49Data[$key] + $ageBetween50And64Data[$key] + $ageMoreThan65Data[$key];
                    $totalDataAllSites += $totalDataBySite[$key];
                    $ageLessThan5Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($ageLessThan5Data[$key] / $totalDataBySite[$key]) * 100)) : 0;
                    $ageBetween5And49Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($ageBetween5And49Data[$key] / $totalDataBySite[$key]) * 100)) : 0;
                    $ageBetween50And64Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($ageBetween50And64Data[$key] / $totalDataBySite[$key]) * 100)) : 0;
                    $ageMoreThan65Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($ageMoreThan65Data[$key] / $totalDataBySite[$key]) * 100)) : 0;

                    $totalPercentPerSite[$key] = $ageLessThan5Data[$key] + $ageBetween5And49Data[$key] + $ageBetween50And64Data[$key] + $ageMoreThan65Data[$key];

                    // init the biggest value to the first value then we're going to check the rest of variables.
                    $biggest = $ageLessThan5Data[$key];
                    if ($ageBetween5And49Data[$key] > $biggest) {
                        $biggest = $ageBetween5And49Data[$key];
                    }
                    if ($ageBetween50And64Data[$key] > $biggest) {
                        $biggest = $ageBetween50And64Data[$key];
                    }
                    if ($ageMoreThan65Data[$key] > $biggest) {
                        $biggest = $ageMoreThan65Data[$key];
                    }
                    // here we are going to check if the total of percent per site
                    // if equal to 99 we will add +1 to the biggest value in order to get 100% in the total of percent per site
                    if ($totalPercentPerSite[$key] == 99) {
                        switch ($biggest) {
                            case $ageLessThan5Data[$key]: $ageLessThan5Data[$key] += 1; break;
                            case $ageBetween5And49Data[$key]: $ageBetween5And49Data[$key] += 1; break;
                            case $ageBetween50And64Data[$key]: $ageBetween50And64Data[$key] += 1; break;
                            case $ageMoreThan65Data[$key]: $ageMoreThan65Data[$key] += 1; break;
                        }
                    }
                    // here we are going to check if the total of percent per site
                    // if equal to 101 we will substr -1 to the biggest value in order to get 100% in the total of percent per site
                    if ($totalPercentPerSite[$key] == 101) {
                        switch ($biggest) {
                            case $ageLessThan5Data[$key]: $ageLessThan5Data[$key] -= 1; break;
                            case $ageBetween5And49Data[$key]: $ageBetween5And49Data[$key] -= 1; break;
                            case $ageBetween50And64Data[$key]: $ageBetween50And64Data[$key] -= 1; break;
                            case $ageMoreThan65Data[$key]: $ageMoreThan65Data[$key] -= 1; break;
                        }
                    }
                }


            }
            return array(
                'sitesChart' => $sites,
                'totalDataAllSites' => $totalDataAllSites,
                'totalDataBySite' => $totalDataBySite,
                'ageLessThan5' => $ageLessThan5Data,
                'ageBetween5And49' => $ageBetween5And49Data,
                'ageBetween50And64' => $ageBetween50And64Data,
                'ageMoreThan65' => $ageMoreThan65Data,
            );
        }
    }

    public function querySQLView5($like, $season, $virus, $site): array
    {
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $querySQL = 'SELECT COUNT(tranche_age_def3) AS tranche_age_def3, zone FROM (SELECT (d.tranche_age_def3) AS tranche_age_def3, s.zone, s.id
                            FROM data_dashboard d 
                            LEFT JOIN site s ON d.site_id = s.id
                            WHERE d.tranche_age_def3 LIKE "' . $like . '" AND s.zone IN ("' . implode("\",\"", $site) . '")';
        if (!empty($virus) && $virus !== "all") {
            if ($virus === "orv_plus") {
                $querySQL .= ' AND (d.' . $virus . ' = 1 OR d.both_positive_result = 1) ';
            } else {
                $querySQL .= ' AND d.' . $virus . ' = 1 ';
            }
        }
        $querySQL .= 'AND d.season ="' . $season . '") AS tmp GROUP BY zone;';
        return $connDB->fetchAll($querySQL);
    }
    // END code View 5 Distribution of patients by age group


    // start code View 6 PCP
    /**
     * @Route("/init-data-view6", name="init_data_view6_ajax")
     */
    public function initDataView6(Request $request): JsonResponse
    {
        $data = $request->get('data');
        $season = $data['season'];
        // get list of sites valid in the selected season ( at least they have row in table data)
        $sites = $this->getValidSitesBySeason($season);
        // get data View6
        $dataView6 = $this->getDataView6($season, $sites);

        return new JsonResponse([
            'status' => 200,
            'current_season' => $season,
            //DATA FOR THE CURVE Distribution LCI
            'comborbility' => $dataView6['comborbility'],
            'oneComborbility' => $dataView6['oneComborbility'],
            'twoOreMoreComborbilities' => $dataView6['twoOreMoreComborbilities'],
            'sites' => $dataView6['sites'],
            'sitesChart' => $dataView6['sites'],
            'totalDataAllSite' => $dataView6['totalDataAllSite'],
            'totalDataBySite' => $dataView6['totalDataBySite']
            // END DATA CURVE
        ]);
    }

    /**
     * @Route("/statistics-view6", name="statistics_view6_ajax")
     */
    public function dataStatisticsView6(Request $request): JsonResponse
    {
        // GET DATA FROM THE FORMA DATA IN AJAX
        $data = $request->get('data');
        //LIST OF SEASONS SELECTED
        $season = $data['season'];
        // LIST Of virus SELECTED
        $virus = $data['virus'];
        $sites = $this->getValidSitesBySeason($season);
        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataView6($season, $sites, $virus)]);
    }

    public function listKeysSites($sites): array
    {
        $sitesKeys = [];
        // GET ALL LIST OF SITES
        $sitesResult = $this->listSites();
        foreach ($sitesResult as $site) {
            if (in_array($site["zone"], $sites)) {
                $sitesKeys[] = $site["zone"];
            }
        }
        return $sitesKeys;
    }

    public function getDataView6($season, $sites, $virus = null)
    {
        //  LIST OF SITES BY KEYS
        $sitesKeys = $this->listKeysSites($sites);
        // intialize the result in array format and all data by site to 0
        $totalDataBySite = $comborbilityData = $oneComborbilityData =
        $twoOreMoreComborbilitiesData = array_combine(array_keys($sitesKeys), array_fill(0, count($sitesKeys), 0));
        $totalDataAllSite = 0;

        if ($season) {
            $comborbility = $this->querySQLView6($season, $virus, "= 0");
            $oneComborbility = $this->querySQLView6($season, $virus, "= 1");
            $twoOreMoreComborbilities = $this->querySQLView6($season, $virus, "> 1");


            // data comborbility
            foreach ($comborbility as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $comborbilityData[$key] = $item["total"];
            }
            // data oneComborbility
            foreach ($oneComborbility as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $oneComborbilityData[$key] = $item["total"];
            }

            // data  twoOreMoreComborbilities
            foreach ($twoOreMoreComborbilities as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $twoOreMoreComborbilitiesData[$key] = $item["total"];
            }

            $listSites = [];
            foreach ($sites as $item) {
                $key = array_search($item, $sitesKeys);
                $totalDataBySite[$key] = $comborbilityData[$key] + $oneComborbilityData[$key] + $twoOreMoreComborbilitiesData[$key];
                if ($totalDataBySite[$key] != 0) {
                    $listSites[] = $item;
                } elseif ($totalDataBySite[$key] == 0) {
                    unset($totalDataBySite[$key]);
                    unset($comborbilityData[$key]);
                    unset($oneComborbilityData[$key]);
                    unset($twoOreMoreComborbilitiesData[$key]);
                }
            }
            // rebuild result to match new indexs and keys
            $totalDataBySite = array_values($totalDataBySite);
            $comborbilityData = array_values($comborbilityData);
            $oneComborbilityData = array_values($oneComborbilityData);
            $twoOreMoreComborbilitiesData = array_values($twoOreMoreComborbilitiesData);

            //get new keys of new list of sites
            $listKeys = $this->listKeysSites($listSites);

            // calcul of % for each site en function of sum total of the 3 variable
            foreach ($listSites as $item) {
                $key = array_search($item, $listKeys);
                $totalDataBySite[$key] = $comborbilityData[$key] + $oneComborbilityData[$key] + $twoOreMoreComborbilitiesData[$key];
                $totalDataAllSite += $totalDataBySite[$key];
                $comborbilityData[$key] = $totalDataBySite[$key] > 0 ? intval(round(($comborbilityData[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $oneComborbilityData[$key] = $totalDataBySite[$key] > 0 ? intval(round(($oneComborbilityData[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $twoOreMoreComborbilitiesData[$key] = $totalDataBySite[$key] > 0 ? intval(round(($twoOreMoreComborbilitiesData[$key] / $totalDataBySite[$key]) * 100)) : 0;

                $totalPercentPerSite[$key] = $comborbilityData[$key] + $oneComborbilityData[$key] + $twoOreMoreComborbilitiesData[$key];

                //init the biggest value to the first value
                // then we are going to verify all the rest of variables to get the biggest value
                $biggest = $comborbilityData[$key];
                if ($oneComborbilityData[$key] > $biggest) {
                    $biggest = $oneComborbilityData[$key];
                }
                if ($twoOreMoreComborbilitiesData[$key] > $biggest) {
                    $biggest = $twoOreMoreComborbilitiesData[$key];
                }

                // here we are going to check if the total of percent per site
                // if equal to 99 we will add +1 to the biggest value in order to get 100% in the total of percent per site
                if ($totalPercentPerSite[$key] == 99) {
                    switch ($biggest) {
                        case $comborbilityData[$key]:
                            $comborbilityData[$key] += 1;
                            break;
                        case $oneComborbilityData[$key]:
                            $oneComborbilityData[$key] += 1;
                            break;
                        case $twoOreMoreComborbilitiesData[$key]:
                            $twoOreMoreComborbilitiesData[$key] += 1;
                            break;
                    }
                }
                // here we are going to check if the total of percent per site
                // if equal to 100 we will substr -1 to the biggest value in order to get 100% in the total of percent per site
                if ($totalPercentPerSite[$key] == 101) {
                    switch ($biggest) {
                        case $comborbilityData[$key]:
                            $comborbilityData[$key] -= 1;
                            break;
                        case $oneComborbilityData[$key]:
                            $oneComborbilityData[$key] -= 1;
                            break;
                        case $twoOreMoreComborbilitiesData[$key]:
                            $twoOreMoreComborbilitiesData[$key] -= 1;
                            break;
                    }
                }
            }

            return array(
                'sitesChart' => $listSites,
                'sites' => $listSites,
                'totalDataAllSite' => $totalDataAllSite,
                'totalDataBySite' => $totalDataBySite,
                'comborbility' => $comborbilityData,
                'oneComborbility' => $oneComborbilityData,
                'twoOreMoreComborbilities' => $twoOreMoreComborbilitiesData,
            );
        }
    }

    public function querySQLView6($season, $virus, $condition): array
    {
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);
        $querySQL = 'SELECT site.zone, COUNT(*) as total
                     FROM data_dashboard d
                     LEFT JOIN site
                     on d.site_id = site.id
                     WHERE ((IF(d.both_chronic_cvd_blood = "1" OR d.both_chronic_cvd_blood = 1, 1, 0)) + 
                           (IF(d.both_chronic_copd = "1" OR d.both_chronic_copd = 1, 1, 0)) + 
                           (IF(d.both_chronic_asthma = "1" OR d.both_chronic_asthma = 1, 1, 0)) + 
                           (IF(d.both_chronic_diabetes = "1" OR d.both_chronic_diabetes = 1, 1, 0)) + 
                           (IF(d.both_chronic_immuno = "1", 1, 0)) + 
                           (IF(d.both_chronic_renal = "1" OR d.both_chronic_renal = 1, 1, 0)) + 
                           (IF(d.both_chronic_rheumatologic_autoimmune_dis = "1" OR d.both_chronic_rheumatologic_autoimmune_dis = 1, 1, 0)) + 
                           (IF(d.both_chronic_neurological_neuromuscular_dis = "1" OR d.both_chronic_neurological_neuromuscular_dis = 1, 1, 0)) + 
                           (IF(d.both_chronic_cirrhosis_liver_dis = "1" OR d.both_chronic_cirrhosis_liver_dis = 1, 1, 0)) + 
                           (IF(d.both_chronic_neoplasm = "1" OR d.both_chronic_neoplasm = 1, 1, 0)) + 
                           (IF(d.both_chronic_obesity = "1" OR d.both_chronic_obesity = 1, 1, 0)) + 
                           (IF(d.both_chronic_malnutri = "1" OR d.both_chronic_malnutri = 1, 1, 0)) + 
                           (IF(d.both_chronic_active_tuberculosis = "1" OR d.both_chronic_active_tuberculosis = 1, 1, 0)) + 
                           (IF(d.both_chronic_hiv_infection = "1" OR d.both_chronic_hiv_infection = 1, 1, 0)) + 
                           (IF(d.both_chronic_hiv_exposure = "1" OR d.both_chronic_hiv_exposure = 1, 1, 0)) + 
                           (IF(d.both_chronic_other = "1" OR d.both_chronic_other = 1, 1, 0)) + 
                           (IF(d.both_chronic_prematurity = "1" OR d.both_chronic_prematurity = 1, 1, 0)) + 
                           (IF(d.both_chronic_hemoglobinopathies = "1" OR d.both_chronic_hemoglobinopathies = 1, 1, 0))) ' . $condition;

        if (!empty($virus) && $virus !== "all") {
            if ($virus === "orv_plus") {
                $querySQL .= ' AND (d.' . $virus . ' = 1 OR d.' . $virus . ' = "1" OR d.both_positive_result = 1 OR d.both_positive_result = "1") ';
            } else {
                $querySQL .= ' AND (d.' . $virus . ' = 1 OR d.' . $virus . ' = "1" )';
            }
        }

        $querySQL .= ' AND d.season ="' . $season . '" GROUP BY site.zone order by CASE 
                              WHEN site.order_zone IS NULL THEN 1 
                            ELSE 0 
                            END,
                      site.order_zone ASC;';
        return $connDB->fetchAll($querySQL);
    }

    //End view 6


    // start Code View 7
    /**
     * @Route("/init-data-view7", name="init_data_view7_ajax")
     */
    public function initDataView7(Request $request): JsonResponse
    {
        $data = $request->get('data');
        $season = $data['season'];
        // get list of sites valid in the selected season ( at least they have row in table data)
        $listSitesHasDataInSelectedSeason = $this->getValidSitesBySeason($season);

        // get data Distribution of lci by season
        $dataView7 = $this->getDataView7($season);

        return new JsonResponse([
            'status' => 200,
            'bar1' => $dataView7['bar1'],
            'bar2' => $dataView7['bar2'],
            'bar3' => $dataView7['bar3'],
            'percent' => $dataView7['percent'],
            'labelBar1' => $dataView7['labelBar1'],
            'sites' => $listSitesHasDataInSelectedSeason
        ]);
    }

    /**
     * @Route("/statistics-view7", name="statistics_view7_ajax")
     */
    public function dataStatisticsView7(Request $request): JsonResponse
    {
        // GET DATA FROM THE FORMA DATA IN AJAX
        $data = $request->get('data');
        //LIST OF SEASONS SELECTED
        $season = $data['season'];
        $site = $data['site'];
        $virus = $data['virus'];

        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataView7($season, $site, $virus),
            'sites' => $this->getValidSitesBySeason($season)
        ]);
    }

    public function getDataView7($season, $site = null, $virus = null)
    {
        // initialize the result in array format and all data by site to 0
        $bar1Data = $bar2Data = $bar3Data = [0];
        $labelBar1 = '#Enrolled';
        if ($season) {
            if ($virus == "both_positive_result") {
                $bar1 = $this->querySQLVUE7('both_positive_result', $season, $site, $virus);
                $bar2 = $this->querySQLVUE7('both_icu_adm', $season, $site, $virus, "both_positive_result");
                $bar3 = $this->querySQLVUE7('both_death_hosp', $season, $site, $virus, "both_positive_result");
                $bar1Data[0] = $this->getTotal($bar1, "both_positive_result");
                $labelBar1 = '#Influenza+';
            } elseif ($virus == "both_lab2_sarscov2_yesno") {
                $bar1 = $this->querySQLVUE7('both_lab2_sarscov2_yesno', $season, $site, $virus);
                $bar2 = $this->querySQLVUE7('both_icu_adm', $season, $site, $virus, "both_lab2_sarscov2_yesno");
                $bar3 = $this->querySQLVUE7('both_death_hosp', $season, $site, $virus, "both_lab2_sarscov2_yesno");
                $bar1Data[0] = $this->getTotal($bar1, "both_lab2_sarscov2_yesno");
                $labelBar1 = '#SARS-CoV-2';
            } elseif ($virus == "both_lab2_rs_yesno") {
                $bar1 = $this->querySQLVUE7('both_lab2_rs_yesno', $season, $site, $virus);
                $bar2 = $this->querySQLVUE7('both_icu_adm', $season, $site, $virus, "both_lab2_rs_yesno");
                $bar3 = $this->querySQLVUE7('both_death_hosp', $season, $site, $virus, "both_lab2_rs_yesno");
                $bar1Data[0] = $this->getTotal($bar1, "both_lab2_rs_yesno");
                $labelBar1 = '#RSV';
            } elseif ($virus == "orv_plus_sans_rsv_sans_sars_cov_2") {
                $bar1 = $this->querySQLVUE7('orv_plus_sans_rsv_sans_sars_cov_2', $season, $site, $virus);
                $bar2 = $this->querySQLVUE7('both_icu_adm', $season, $site, $virus, "orv_plus_sans_rsv_sans_sars_cov_2");
                $bar3 = $this->querySQLVUE7('both_death_hosp', $season, $site, $virus, "orv_plus_sans_rsv_sans_sars_cov_2 ");
                $bar1Data[0] = $this->getTotal($bar1, "orv_plus_sans_rsv_sans_sars_cov_2");
                $labelBar1 = '#ORV';
            } else {
                $bar1 = $this->querySQLVUE7('enrolled', $season, $site, $virus);
                $bar2 = $this->querySQLVUE7('both_icu_adm', $season, $site, $virus);
                $bar3 = $this->querySQLVUE7('both_death_hosp', $season, $site, $virus);
                $bar1Data[0] = $this->getTotal($bar1, "enrolled");
            }


            // data both_icu_adm
            $totalBar2 = 0;
            foreach ($bar2 as $item) {
                $totalBar2 += (int)$item["both_icu_adm"];
            }
            $bar2Data[0] = $totalBar2;

            // data both_death_hosp
            $totalBar3 = 0;
            foreach ($bar3 as $item) {
                $totalBar3 += (int)$item["both_death_hosp"];
            }
            $bar3Data[0] = $totalBar3;
            $percentBars = [$this->calculatePercentageView7($season, $site, "both_icu_adm", $virus),
                $this->calculatePercentageView7($season, $site, "both_death_hosp", $virus)];
            return array(
                'bar1' => $bar1Data,
                'bar2' => $bar2Data,
                'bar3' => $bar3Data,
                'percent' => $percentBars,
                'labelBar1' => $labelBar1
            );
        }
    }

    public function getTotal($result, $variable): int
    {
        $total = 0;
        foreach ($result as $item) {
            $total += (int)$item[$variable];
        }
        return $total;
    }

    public function querySQLVUE7($variable, $season, $site, $virus, $variable2 = null): array
    {
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        if ($variable == "enrolled") {
            $querySQL = 'SELECT COUNT(*) AS ' . $variable . '
                         FROM data_dashboard d
                         LEFT JOIN site s ON d.site_id = s.id
                         WHERE d.season = "' . $season . '"';
            if (!empty($site) && $site != "all") {
                $querySQL .= ' AND  s.zone = "' . $site . '"';
            }
            return $connDB->fetchAll($querySQL);
        } else {
            $querySQL = 'SELECT COUNT(' . $variable . ') AS ' . $variable . ', zone FROM (SELECT (d.' . $variable . ') AS ' . $variable . ', s.zone, s.id
                            FROM data_dashboard d
                            LEFT JOIN site s ON d.site_id = s.id
                            WHERE d.' . $variable . ' = 1';
            if ($variable2) {
                $querySQL .= ' AND  d.' . $variable2 . ' = 1';
            }
            if (!empty($site) && $site != "all") {
                $querySQL .= ' AND  s.zone = "' . $site . '"';
            }
            if (!empty($virus) && $virus != "all") {
                $querySQL .= ' AND  d.' . $virus . ' = 1';
            }
            $querySQL .= ' AND d.season = "' . $season . '") AS tmp GROUP BY zone;';
            return $connDB->fetchAll($querySQL);
        }


    }

    public function calculatePercentageView7($season, $site, $variable, $virusFilter = 'all'): string
    {
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $baseQuery = 'SELECT SUM(' . $variable . ' = 1) AS ' . $variable . ',
                             SUM(both_positive_result = 1) AS both_positive_result,
                             SUM(both_lab2_sarscov2_yesno = 1) AS both_lab2_sarscov2_yesno,
                             SUM(both_lab2_rs_yesno = 1) AS both_lab2_rs_yesno,
                             SUM(orv_plus_sans_rsv_sans_sars_cov_2 = 1) AS orv_plus_sans_rsv_sans_sars_cov_2,
                             SUM(both_positive_result = 1 OR orv_plus = 1) AS ' . $variable . '1,
                             SUM(' . $variable . ' = 1 AND both_positive_result = 1) AS ' . $variable . '2,
                             SUM(' . $variable . ' = 1 AND both_lab2_sarscov2_yesno = 1) AS ' . $variable . '3,
                             SUM(' . $variable . ' = 1 AND both_lab2_rs_yesno = 1) AS ' . $variable . '4,
                             SUM(' . $variable . ' = 1 AND orv_plus_sans_rsv_sans_sars_cov_2 = 1) AS ' . $variable . '5
                  FROM data_dashboard d
                  LEFT JOIN site s ON d.site_id = s.id
                  WHERE d.season ="' . $season . '"';

        if (!empty($site) && $site != "all") {
            $baseQuery .= ' AND  s.zone = "' . $site . '"';
        }
        switch ($virusFilter) {
            case 'both_positive_result':
                $querySQL = $baseQuery . ' AND both_positive_result = 1';
                break;
            case 'both_lab2_sarscov2_yesno':
                $querySQL = $baseQuery . ' AND both_lab2_sarscov2_yesno = 1';
                break;
            case 'both_lab2_rs_yesno':
                $querySQL = $baseQuery . ' AND both_lab2_rs_yesno = 1';
                break;
            case 'orv_plus_sans_rsv_sans_sars_cov_2':
                $querySQL = $baseQuery . ' AND orv_plus_sans_rsv_sans_sars_cov_2 = 1';
                break;
            default:
                $querySQL = $baseQuery;
                break;
        }

        $stmt = $connDB->prepare($querySQL);
        $stmt->execute();
        $result = $stmt->fetch();

        // caalculate the percentage based on the filter
        $percentage = 0;
        switch ($virusFilter) {
            case 'both_positive_result':
                if ($result['both_positive_result'] != 0) {
                    $percentage = round((($result["$variable" . "2"] * 100) / $result['both_positive_result']));
                }
                break;
            case 'both_lab2_sarscov2_yesno':
                if ($result['both_lab2_sarscov2_yesno'] != 0) {
                    $percentage = round((($result["$variable" . "3"] * 100) / $result['both_lab2_sarscov2_yesno']));
                }
                break;
            case 'both_lab2_rs_yesno':
                if ($result['both_lab2_rs_yesno'] != 0) {
                    $percentage = round((($result["$variable" . "4"] * 100) / $result['both_lab2_rs_yesno']));
                }
                break;
            case 'orv_plus_sans_rsv_sans_sars_cov_2':
                if ($result['orv_plus_sans_rsv_sans_sars_cov_2'] != 0) {
                    $percentage = round((($result["$variable" . "5"] * 100) / $result['orv_plus_sans_rsv_sans_sars_cov_2']));
                }
                break;
            default:
                if ($result["$variable" . "1"] != 0) {
                    $percentage = round((($result["$variable"] * 100) / $result["$variable" . "1"]));
                }
                break;

        }
        return $percentage;
    }

    // End Code View 7


    /**
     * Description :: add export airfinity user to db
     * @Route("/add_export_dashboard_user", name="add_export_dashboard_user")
     */
    public function addExportDashboardUser(Request $request): JsonResponse
    {
        $data = $request->request->get('data');
        try {

            $dashboardUserExport = new DashboardUserExport();
            $dashboardUserExport->setEmailAdress($data['emailaddress']);
            $dashboardUserExport->setFullname($data['fullname']);
            $dashboardUserExport->setInstitution($data['institution']);
            $dashboardUserExport->setCountry($data['country']);
            $dashboardUserExport->setGraphNumber($data['graphNumber']);
            $dashboardUserExport->setGraphName($data['graphName']);
            $dashboardUserExport->setIsUsed(false);
            $dashboardUserExport->setCreatedAt(new \DateTime());

            $this->em->persist($dashboardUserExport);
            $this->em->flush();

            // url du fichier
            $url = $request->getSchemeAndHttpHost() . "/dashboard/download/excel/" . $dashboardUserExport->getToken();

            //send email
            $this->emailsTemplateSender->email_send_dashboard_user_export($data['emailaddress'], $data['fullname'], $url);

            return new JsonResponse([
                'status' => 200,
                'graph' => $data['graphName'],
                'message' => 'An email will be sent when the export is ready'
            ]);

        } catch (Exception $e) {
            return new JsonResponse([
                'status' => 500,
                'message' => "Something is went wrong."
            ]);
        }
    }

    /**
     * Description :: downlad excel
     * @Route("/download/excel/{token}", name="download_excel")
     */
    public function downloadExcel(string $token): Response
    {
        // Rechercher l'entité par le token
        $dashboardUserExport = $this->em->getRepository(DashboardUserExport::class)->findOneBy(['token' => $token]);

        if (!$dashboardUserExport) {
            return $this->render('dashboard/index.html.twig', [
                'error' => "Token invalid!",
                'listSites' => $this->listSites(),
                'categories' => $this->listSitesCategory(),
                'countries' => Countries::getNames()
            ]);
        }

        // Vérifier si le token a expiré
        if (new \DateTime() > $dashboardUserExport->getExpiresAt()) {
            return $this->render('dashboard/index.html.twig', [
                'error' => "The token has expired!",
                'listSites' => $this->listSites(),
                'categories' => $this->listSitesCategory(),
                'countries' => Countries::getNames()
            ]);
        }

        // Vérifier si le token a expiré
        if ($dashboardUserExport->getIsUsed()) {
            return $this->render('dashboard/index.html.twig', [
                'error' => "The token has been used!",
                'listSites' => $this->listSites(),
                'categories' => $this->listSitesCategory(),
                'countries' => Countries::getNames()
            ]);
        }

        $graphName = $dashboardUserExport->getGraphName();

        // Chemin du fichier
        $filePath = $this->getParameter('kernel.project_dir') . '/public/uploads/dashboard_export/' . $graphName . '.xlsx';


        if (!file_exists($filePath)) {
            return $this->render('dashboard/index.html.twig', [
                'error' => "The file does not exist!",
                'listSites' => $this->listSites(),
                'categories' => $this->listSitesCategory(),
                'countries' => Countries::getNames()
            ]);
        }

        // Créer la réponse pour le téléchargement
        $response = new BinaryFileResponse($filePath);
        $response->setContentDisposition(
            ResponseHeaderBag::DISPOSITION_ATTACHMENT,
            basename($filePath)
        );

        $dashboardUserExport->setIsUsed(true);
        $dashboardUserExport->setUpdatedAt(new \DateTime());
        $this->em->persist($dashboardUserExport);
        $this->em->flush();

        return $response;
    }
}
