<?php

namespace App\Controller;

use App\Entity\Main\Data;
use App\Entity\Main\Site;
use App\Service\AppParams;
use App\Service\EmailsTemplateSender;
use App\Service\Functions;
use App\Service\GlobalServices;
use Doctrine\DBAL\Driver\Connection;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Container\ContainerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

class StatisticsController extends AbstractController
{
    protected $em;
    protected $globalServices;

    public function __construct(EntityManagerInterface $entityManager,
                                GlobalServices         $globalServices
    )
    {
        $this->em = $entityManager;
        $this->globalServices = $globalServices;
    }

    /**
     * @Route("/statistics", name="statistics_page")
     */
    public function index()
    {
        // get the current season and date start of season
        $result = $this->getCurrentSeason();
        $startSeason = $result['startDateSeason'];

        return $this->render('statistics/index.html.twig', [
//            // LIST OF ALL SEASON IN OUR DB
            'seasons' => $this->em->getRepository(Data::class)->seasonStatistics(),
            // LIST OF ALL SITES IN OUR DB
            'listSites' => $this->listSites(),
//            // LIST OF ALL WEEK NUMBERS FROM START OF CURRENT SEASON UNTIL DATE SYSTEM
            'number_week' => $this->getAllNumberWeeksSeason($startSeason),
        ]);
    }

    // start block of distribution of viruses

    /**
     * @Route("/init-data-distribution-viruses", name="init_data_distribution_viruses_ajax")
     */
    public function initDataDistributionViruses()
    {
        // get the current season and date start of season
        $result = $this->getCurrentSeason();
        $season = $result['season'];
        // get list of sites valid in the selected season ( at least they have row in table data)
        $listSitesHasDataInSelectedSeason = $this->getlistValidSitesBySeason(explode(',', $season));

        // get data Distribution of Viruses by season
        $dataDistributionViruses = $this->getDataDistributionViruses(explode(',', $season), $listSitesHasDataInSelectedSeason);

        return new JsonResponse([
            'status' => 200,
            'current_season' => $season,
            //DATA FOR THE FIRST CURVE Distribution of Viruses
            'sites' => $dataDistributionViruses['sites'],
            'enrolled' => $dataDistributionViruses['enrolled'],
            'totalIncluded' => $dataDistributionViruses['totalIncluded'],
            'influenza' => $dataDistributionViruses['influenza'],
            'SARSCoV2' => $dataDistributionViruses['SARSCoV2'],
            'EVD68' => $dataDistributionViruses['EVD68'],
            'RSV' => $dataDistributionViruses['RSV'],
            'ORV' => $dataDistributionViruses['ORV'],
            // END DATA FIRST CURVE
        ]);
    }

    /**
     * Description: get data of distribution of viruses when the user make a filter
     * @Route("/statistics-distribution-viruses", name="statistics_distribution_viruses_ajax")
     */
    public function dataStatisticsDistributionViruses(Request $request)
    {
        // GET DATA FROM THE FORMA DATA IN AJAX
        $data = $request->get('data');
        //LIST OF SEASONS SELECTED
        $seasons = $data['seasons'];
        // LIST Of SITES SELECTED
        $sites = $data['sites'];

        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataDistributionViruses(explode(',', $seasons), explode(',', $sites))]);
    }

    // Global function we call to return the data of distribution of viruses
    // we give it the list of seasons and the list of sites as parameters
    public function getDataDistributionViruses($season, $sites)
    {
        /* connexion à la base de données */
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        //  LIST OF SITES BY KEYS
        $sitesKeys = $this->listKeysSites($sites);
        // intialize the result in array format and all data by site to 0
        $enrolledData = $influenzaData = $SARSCoV2Data = $RSVData = $ORVData = $EVD68Data = array_combine(array_keys($sitesKeys), array_fill(0, count($sitesKeys), 0));

        if ($season) {
            //data enrolled
            $totalIncluded = 0;
            $enrolled = $this->sqlQueryDV($season, $sites, 'included');
            foreach ($enrolled as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $enrolledData[$key] = (int)$item["included"];
                $totalIncluded += (int)$item["included"];
            }

            //data Influenza
            $influenza = $this->sqlQueryDV($season, $sites, 'lci_plus');
            foreach ($influenza as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $influenzaData[$key] = (int)$item["lci_plus"];
            }

            //data SARSCoV2
            $SARSCoV2 = $this->sqlQueryDV($season, $sites, 'both_lab2_sarscov2_yesno');
            foreach ($SARSCoV2 as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $SARSCoV2Data[$key] = (int)$item["both_lab2_sarscov2_yesno"];
            }

            //data EV D68
            $EVD68 = $this->sqlQueryDV($season, $sites, 'both_lab2_evd68_yesno');
            foreach ($EVD68 as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $EVD68Data[$key] = (int)$item["both_lab2_evd68_yesno"];
            }

            //data RSV
            $RSV = $this->sqlQueryDV($season, $sites, 'both_lab2_rs_yesno');
            foreach ($RSV as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $RSVData[$key] = (int)$item["both_lab2_rs_yesno"];
            }

            //data ORV
            $ORV = $this->sqlQueryDV($season, $sites, 'orv_plus_sans_rsv_sans_sars_cov_2');
            foreach ($ORV as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $ORVData[$key] = (int)$item["orv_plus_sans_rsv_sans_sars_cov_2"];
            }
        }
        return [
            'sites' => $sitesKeys,
            'enrolled' => $enrolledData,
            'totalIncluded' => $totalIncluded,
            'influenza' => $influenzaData,
            'SARSCoV2' => $SARSCoV2Data,
            'EVD68' => $EVD68Data,
            'RSV' => $RSVData,
            'ORV' => $ORVData,
        ];
    }

    // global function to get data of curve distribution of viruses
    private function sqlQueryDV($season, $sites, $variable)
    {

        /* connexion à la base de données */
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $sqlQuery = 'SELECT SUM(' . $variable . ') AS ' . $variable . ', zone FROM (SELECT COUNT(d.' . $variable . ') AS ' . $variable . ', d.season, s.id, s.zone
                     FROM data d 
                     LEFT JOIN site s ON d.site_id = s.id
                     WHERE d.' . $variable . ' = 1 AND d.season IN ("' . implode("\",\"", $season) . '") AND s.zone IN ("' . implode("\",\"", $sites) . '")
                     GROUP BY d.season, s.zone ) AS tmp GROUP BY id ORDER BY zone;';
        $reaultQuery = $connDB->fetchAll($sqlQuery);

        return $reaultQuery;
    }

    // End block distribution of viruses

    // start global function for all curves
    /**
     * description : a function we called in ajax for each change in the input season,we give it as params the list
     * of season selected and return the valid list of sites
     * @Route("/statistics-valid-site-distribution", name="valid_site_distribution_ajax")
     */
    public function listValidSites(Request $request)
    {
        return new JsonResponse([
            'status' => 200,
            'valid_sites' => $this->getlistValidSitesBySeason(explode(',', $request->get('data')['seasons'])),
            'valid_weeks' => $this->getAllNumberWeeksBySeason(explode(',', $request->get('data')['seasons']))
        ]);
    }

    // get list of sites at least they have row in table data
    public function getlistValidSitesBySeason($season)
    {
        // connect to the database
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);
        // requet sql select distinct sites in alphabetical order and at least they have row in table data
        $sqlListSites = 'SELECT DISTINCT(s.zone) FROM data d LEFT JOIN site s on d.site_id = s.id WHERE d.season IN ("' . implode("\",\"", $season) . '") ORDER BY s.zone ASC;';
        $sites = $connDB->fetchAll($sqlListSites);
        $listSites = [];
        foreach ($sites as $site) {
            array_push($listSites, $site["zone"]);
        }
        // returns an array containing the list of sites valid
        return $listSites;
    }

    // get list of weeks numbers by season
    public function getlistWeeksNumberBySeason($season)
    {
        // connect to the database
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);
        // requet sql select distinct sites in alphabetical order and at least they have row in table data
        $sqlListWeeksNumber = 'SELECT DISTINCT(week_number) AS week_number FROM data  WHERE season ="' . $season . '" ';
        $result = $connDB->fetchAll($sqlListWeeksNumber);
        $listWeeksNumber = [];
        foreach ($result as $week) {
            array_push($listWeeksNumber, $week["week_number"]);
        }
        // returns an array containing the list of Week Number valid
        return $listWeeksNumber;
    }

    /**
     * @Route("/statistics-valid-week-number-distribution-influenza", name="valid_week_number_distribution_influenza_ajax")
     */
    public function getlistValidWeeksNumberBySites(Request $request)
    {

        // get the current season and date start of season
        $result = $this->getCurrentSeason();
        $season = $result['season'];

        // connect to the database
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);
        // requet sql select distinct sites in alphabetical order and at least they have row in table data
        $sqlListWeeksNumber = 'SELECT DISTINCT(week_number) AS week_number 
                               FROM data d LEFT JOIN site s ON s.id = d.site_id 
                               WHERE season IN ("' . implode("\",\"", explode(',', $request->get('data')['seasons'])) . '") AND s.zone IN ("' . implode("\",\"", explode(',', $request->get('data')['sites'])) . '");';
        $result = $connDB->fetchAll($sqlListWeeksNumber);
        $listWeeksNumber = [];
        foreach ($result as $week) {
            array_push($listWeeksNumber, $week["week_number"]);
        }
        // returns an array containing the list of Week Number valid
        return new JsonResponse([
            'status' => 200,
            'valid_weeks' => $listWeeksNumber
        ]);
    }

    // get season from the current date
    public function getCurrentSeason()
    {
        $currentDate = new \DateTime(); // Get the current date in YYYY-MM-DD format
        $fullYear = $currentDate->format('Y'); // format YYYY
        $year = $currentDate->format('y'); // format YY
        // Create a November 1 date for the given year
        $november1 = date('Y-m-d', strtotime("{$fullYear}-11-01"));
        $date = $currentDate->format('Y-m-d');
        if ($date < $november1) {
            // The current date is before November 1 of the given year, so it's in the current year-1_year
            $season = $currentDate->modify("-1 year")->format('y') . "_" . date($year);
            $startDate = date('Y-m-d', strtotime("{$currentDate->format('Y')}-11-01"));
        } else {
            // The current date is on or after November 1 of the given year, so it's in the current year_year+1
            $season = date($year) . "_" . $currentDate->modify("+1 year")->format('y');
            $startDate = $november1;
        }
        return array(
            'season' => $season,
            'startDateSeason' => $startDate
        );
    }

    // get all list sites from database in alphabetical order (order by zone ASC)
    public function listSites()
    {
        //  connect to the database
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);
        // requet sql to select all site from table site order by zone
        $sqlListSites = "SELECT zone FROM site ORDER BY zone ASC";
        $sites = $connDB->fetchAll($sqlListSites);
        // return list site in an array
        return $sites;
    }

    // get list of weeks number for the current season
    public function getAllNumberWeeksSeason($startSeason)
    {
        // Set the start date to November 1, date of start season
        $startDate = new \DateTime($startSeason);
        // Set the end date to the current date
        $endDate = new \DateTime();
        // Create an empty array to store the week numbers
        $weekNumbers = [];
        // Iterate over each week in the date range and add the week number to the array
        while ($startDate <= $endDate) {
            // Get the week number and concat with --Year(YYYY)
            $weekNumbers[] = $startDate->format('W') . '--' . $startDate->format('Y');
            // add a week to the date
            $startDate->modify('+1 week');
        }
        return $weekNumbers;
    }

    // get list of weeks number for a given season or array of seasons
    public function getAllNumberWeeksBySeason($seasons)
    {
        $weekNumbers = [];
        // the current date
        $currentDate = new \DateTime();
        foreach ($seasons as $season) {
            // Extract start and end years from the season parameter
            list($startYear, $endYear) = explode('_', $season);
            // Set the start date to November 1, date of start season
            $startDate = new \DateTime($startYear . '-11-01');
            // Set the end date to October 30, date of end season
            $endDate = new \DateTime(($endYear + 1) . '-10-30');
            // Iterate over each week in the date range and add the week number to the array
            while ($startDate <= $endDate && $startDate <= $currentDate) {
                // Get the week number and concat with --Year(YYYY)
                $weekNumbers[] = $startDate->format('W') . '--' . $startDate->format('Y');
                // add a week to the date
                $startDate->modify('+1 week');
            }
        }
        return $weekNumbers;
    }

    // GLOBAL FUNTION RETURN LIST OF SITES AS KEYS
    private function listKeysSites($sites)
    {
        $sitesKeys = [];
        // GET ALL LIST OF SITES
        $sitesResult = $this->listSites();
        // HERE WE GONNA VERIFY IF THE SITE SELECTED IS EXIST IN ALL LISTS OF SITES
        // IF TRUE WE PUSH IN OUR ARRAY (SITESkYES) THE NAME OF SITE AS KEY
        foreach ($sitesResult as $site) {
            if (in_array($site["zone"], $sites)) {
                array_push($sitesKeys, $site["zone"]);
            }
        }
        return $sitesKeys;
    }

    // END start global function for all curves

    // Start block distribution of influenza
    /**
     * @Route("/init-data-distribution-influenza", name="init_data_distribution_influenza_ajax")
     */
    public function initDataDistributionInfluenzaCurrentSeason()
    {

        // get the current season and date start of season
        $result = $this->getCurrentSeason();
        $season = $result['season'];
        $startSeason = $result['startDateSeason'];
        // Get ALl week numbers by season
        $weeks = $this->getAllNumberWeeksSeason($startSeason);
        // Get the list of valide site in the current season
        $sites = $this->getlistValidSitesBySeason(explode(',', $season));

        // Get list of week valid (at least have a raw in our table data)
        $listWeeksNumber = $this->getlistWeeksNumberBySeason($season);

        $weekNumbersKeys = [];
        foreach ($weeks as $week) {
            if (in_array($week, $listWeeksNumber)) {
                array_push($weekNumbersKeys, $week);
            }
        }

        // get data distribution of Influenza in the current season
        $dataDistributionInfluenza = $this->getDataDistributionInfluenzaCurrentSeason($sites, $weekNumbersKeys, explode(',', $season));

        return new JsonResponse([
            'status' => 200,
            // Date for the second curve Distribution of Influenza in the current season
            'listNumberWeeks' => $weekNumbersKeys,
            'AH1N1pdm09' => $dataDistributionInfluenza['AH1N1pdm09'],
            'AH3' => $dataDistributionInfluenza['AH3'],
            'BYamagataLineage' => $dataDistributionInfluenza['BYamagataLineage'],
            'BVictoriaLineage' => $dataDistributionInfluenza['BVictoriaLineage'],
            'H1NotherLineage' => $dataDistributionInfluenza['H1NotherLineage'],
            'H3NotherLineage' => $dataDistributionInfluenza['H3NotherLineage'],
            'ANotSubtyped' => $dataDistributionInfluenza['ANotSubtyped'],
            'BLineageNotDetermined' => $dataDistributionInfluenza['BLineageNotDetermined'],
            'enrolledDataInfluenza' => $dataDistributionInfluenza['enrolledDataInfluenza'],
            'sites' => $dataDistributionInfluenza['sites'],
            'totalIncluded' => $dataDistributionInfluenza['totalIncluded'],
            'current_season' => $season,
            // END DATA FIRST CURVE
        ]);
    }

    /**
     * @Route("/statistics-distribution-influenza", name="statistics_distribution_influenza_ajax")
     */
    public function dataStatisticsDistributionInfluenzaCurrentSeason(Request $request)
    {
        // get data of sites, weeks and seasons send it from the form
        $data = $request->get('data');
        $seasons = explode(',', $data["seasons"]);
        $sites = explode(',', $data["sites"]);
        $weeks = explode(',', $data["weeks"]);

        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataDistributionInfluenzaCurrentSeason($sites, $weeks, $seasons)]);

    }

    // global function return data for the curve of distribution of influenza on the current season
    public function getDataDistributionInfluenzaCurrentSeason($sites, $weeks, $season)
    {

        // init array result
        $enrolledData = $AH1N1pdm09Data = $AH3Data = $BYamagataLineageData = $BVictoriaLineageData = $H1NotherLineageData = $H3NotherLineageData = $ANotSubtypedData = $BLineageNotDeterminedData = array_combine(array_keys($weeks), array_fill(0, count($weeks), 0));

        //data enrolled
        $enrolled = $this->querySQLDICS($sites, $weeks, $season, 'included');
        $totalIncluded = 0;
        foreach ($enrolled as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $enrolledData[$key] = [$key, (int)$item["included"]];
            $totalIncluded += (int)$item["included"];
        }

        // data AH1N1pdm09
        $AH1N1pdm09 = $this->querySQLDICS($sites, $weeks, $season, 'both_lab_flu_h1n1');
        foreach ($AH1N1pdm09 as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $AH1N1pdm09Data[$key] = (int)$item["both_lab_flu_h1n1"];
        }

        // data AH3
        $AH3 = $this->querySQLDICS($sites, $weeks, $season, 'both_lab_flu_h3n2');
        foreach ($AH3 as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $AH3Data[$key] = (int)$item["both_lab_flu_h3n2"];
        }

        // data  B(Yamagata lineage)
        $BYamagataLineage = $this->querySQLDICS($sites, $weeks, $season, 'both_lab_flu_yamagata');
        foreach ($BYamagataLineage as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $BYamagataLineageData[$key] = (int)$item["both_lab_flu_yamagata"];
        }

        // data  B (Victoria lineage)
        $BVictoriaLineage = $this->querySQLDICS($sites, $weeks, $season, 'both_lab_flu_victoria');
        foreach ($BVictoriaLineage as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $BVictoriaLineageData[$key] = (int)$item["both_lab_flu_victoria"];
        }

        // data  H1Nother
        $H1NotherLineage = $this->querySQLDICS($sites, $weeks, $season, 'both_lab_flu_h1nother');
        foreach ($H1NotherLineage as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $H1NotherLineageData[$key] = (int)$item["both_lab_flu_h1nother"];
        }

        // data  H3Nother
        $H3NotherLineage = $this->querySQLDICS($sites, $weeks, $season, 'both_lab_flu_h3nother');
        foreach ($H3NotherLineage as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $H3NotherLineageData[$key] = (int)$item["both_lab_flu_h3nother"];
        }

        // data A (Not subtyped)
        $ANotSubtyped = $this->querySQLDICS($sites, $weeks, $season, 'both_lab_flu_a_nosub');
        foreach ($ANotSubtyped as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $ANotSubtypedData[$key] = (int)$item["both_lab_flu_a_nosub"];
        }

        // data B (Lineage not determined)
        $sqlBLineageNotDetermined = $this->querySQLDICS($sites, $weeks, $season, 'both_lab_flu_b_nosub');
        foreach ($sqlBLineageNotDetermined as $item) {
            $weekNumber = $item["week_number"];
            $key = array_search($weekNumber, $weeks);
            $BLineageNotDeterminedData[$key] = (int)$item["both_lab_flu_b_nosub"];
        }

        return array(
            'sites' => $sites,
            'listNumberWeeks' => $weeks,
            'enrolledDataInfluenza' => $enrolledData,
            'AH1N1pdm09' => $AH1N1pdm09Data,
            'AH3' => $AH3Data,
            'BYamagataLineage' => $BYamagataLineageData,
            'BVictoriaLineage' => $BVictoriaLineageData,
            'H1NotherLineage' => $H1NotherLineageData,
            'H3NotherLineage' => $H3NotherLineageData,
            'ANotSubtyped' => $ANotSubtypedData,
            'BLineageNotDetermined' => $BLineageNotDeterminedData,
            'totalIncluded' => $totalIncluded
        );
    }

    // GLobal query to get all data of DICS to draw the chart
    //DICS : Distribution of influenza on the current season
    private function querySQLDICS($sites, $weeks, $season, $variable)
    {

        /* connexion à la base de données */
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $sqlQuery = 'SELECT SUM(' . $variable . ') AS ' . $variable . ', week_number FROM (SELECT (d.' . $variable . ') AS ' . $variable . ', d.week_number, s.zone, s.id
                     FROM data d 
                     LEFT JOIN site s ON d.site_id = s.id
                     WHERE d.' . $variable . ' = 1 AND d.week_number IN ("' . implode("\",\"", $weeks) . '") AND s.zone IN ("' . implode("\",\"", $sites) . '") 
                     AND d.season IN ("' . implode("\",\"", $season) . '")) AS tmp GROUP BY week_number;';
        $queryResult = $connDB->fetchAll($sqlQuery);

        return $queryResult;

    }

    // END  block distribution of influenza

    // start  distribution LCI
    // LCI : Distribution of Laboratory-confirmed influenza
    /**
     * @Route("/init-data-distribution-lci", name="init_data_distribution_lci_ajax")
     */
    public function initDataDistributionLCI()
    {
        // get the current season and date start of season
        $result = $this->getCurrentSeason();
        $season = $result['season'];
        // get list of sites valid in the selected season ( at least they have row in table data)
        $listSitesHasDataInSelectedSeason = $this->getlistValidSitesBySeason(explode(',', $season));

        // get data Distribution of lci by season
        $dataDistributionLCI = $this->getDataDistributionLCI(explode(',', $season), $listSitesHasDataInSelectedSeason);

        return new JsonResponse([
            'status' => 200,
            'current_season' => $season,
            //DATA FOR THE CURVE Distribution LCI
            'lci_plus' => $dataDistributionLCI['lci_plus'],
            'totalLciPplus' => $dataDistributionLCI['totalLciPplus'],
            'AH1N1pdm09' => $dataDistributionLCI['AH1N1pdm09'],
            'AH3' => $dataDistributionLCI['AH3'],
            'BYamagataLineage' => $dataDistributionLCI['BYamagataLineage'],
            'BVictoriaLineage' => $dataDistributionLCI['BVictoriaLineage'],
            'H1NotherLineage' => $dataDistributionLCI['H1NotherLineage'],
            'H3NotherLineage' => $dataDistributionLCI['H3NotherLineage'],
            'ANotSubtyped' => $dataDistributionLCI['ANotSubtyped'],
            'BLineageNotDetermined' => $dataDistributionLCI['BLineageNotDetermined'],
            'sites' => $dataDistributionLCI['sites'],
            'sitesChart' => $dataDistributionLCI['sitesChart'],
            'totalPercentPerSite' => $dataDistributionLCI['totalPercentPerSite'],
            // END DATA CURVE
        ]);
    }

    /**
     * Description: get data of distribution of lci when the user make a filter
     * @Route("/statistics-distribution-lci", name="statistics_distribution_lci_ajax")
     */
    public function dataStatisticsDistributionLCI(Request $request)
    {
        // GET DATA FROM THE FORMA DATA IN AJAX
        $data = $request->get('data');
        //LIST OF SEASONS SELECTED
        $seasons = $data['seasons'];
        // LIST Of SITES SELECTED
        $sites = $data['sites'];

        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataDistributionLCI(explode(',', $seasons), explode(',', $sites))]);
    }

    // Global function we call to return the data of distribution of LCI
    // we give it the list of seasons and the list of sites as parameters
    public function getDataDistributionLCI($season, $sites)
    {
        //  LIST OF SITES BY KEYS
        $sitesKeys = $this->listKeysSites($sites);

        // intialize the result in array format and all data by site to 0
        $totalDataBySite = $lciPlusData = $AH1N1pdm09Data = $AH3Data = $BYamagataLineageData = $BVictoriaLineageData = $H1NotherLineageData = $H3NotherLineageData = $ANotSubtypedData = $totalPercentPerSite = $BLineageNotDeterminedData = array_combine(array_keys($sitesKeys), array_fill(0, count($sitesKeys), 0));

        if ($season) {
            $lci_plus = $this->querySQLLCI('lci_plus', $season, $sites);
            $AH1N1pdm09 = $this->querySQLLCI('both_lab_flu_h1n1', $season, $sites);
            $AH3 = $this->querySQLLCI('both_lab_flu_h3n2', $season, $sites);
            $BYamagataLineage = $this->querySQLLCI('both_lab_flu_yamagata', $season, $sites);
            $BVictoriaLineage = $this->querySQLLCI('both_lab_flu_victoria', $season, $sites);
            $H1NotherLineage = $this->querySQLLCI('both_lab_flu_h1nother', $season, $sites);
            $H3NotherLineage = $this->querySQLLCI('both_lab_flu_h3nother', $season, $sites);
            $ANotSubtyped = $this->querySQLLCI('both_lab_flu_a_nosub', $season, $sites);
            $sqlBLineageNotDetermined = $this->querySQLLCI('both_lab_flu_b_nosub', $season, $sites);

            // data lci_plus
            $totalLciPplus = 0;
            foreach ($lci_plus as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $lciPlusData[$key] = (int)$item["lci_plus"];
                $totalLciPplus += (int)$item["lci_plus"];
            }
            // data AH1N1pdm09
            foreach ($AH1N1pdm09 as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $AH1N1pdm09Data[$key] = $item["both_lab_flu_h1n1"];
//                $AH1N1pdm09Data[$key] = number_format((($item["both_lab_flu_h1n1"] / $lciPlusData[$key]) * 100), 2, '.', '');
            }

            // data AH3
            foreach ($AH3 as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $AH3Data[$key] = $item["both_lab_flu_h3n2"];
//                $AH3Data[$key] = number_format((($item["both_lab_flu_h3n2"] / $lciPlusData[$key]) * 100), 2, '.', '');
            }

            // data  B(Yamagata lineage)
            foreach ($BYamagataLineage as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $BYamagataLineageData[$key] = $item["both_lab_flu_yamagata"];
//                $BYamagataLineageData[$key] = number_format((($item["both_lab_flu_yamagata"] / $lciPlusData[$key]) * 100), 2, '.', '');
            }

            // data  B (Victoria lineage)
            foreach ($BVictoriaLineage as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $BVictoriaLineageData[$key] = $item["both_lab_flu_victoria"];
//                $BVictoriaLineageData[$key] = number_format((($item["both_lab_flu_victoria"] / $lciPlusData[$key]) * 100), 2, '.', '');
            }

            foreach ($H1NotherLineage as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $H1NotherLineageData[$key] = $item["both_lab_flu_h1nother"];
            }

            foreach ($H3NotherLineage as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $H3NotherLineageData[$key] = $item["both_lab_flu_h3nother"];
            }

            // data A (Not subtyped)
            foreach ($ANotSubtyped as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $ANotSubtypedData[$key] = $item["both_lab_flu_a_nosub"];
//                $ANotSubtypedData[$key] = number_format((($item["both_lab_flu_a_nosub"] / $lciPlusData[$key]) * 100), 2, '.', '');
            }

            foreach ($sqlBLineageNotDetermined as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $BLineageNotDeterminedData[$key] = $item["both_lab_flu_b_nosub"];
//                $BLineageNotDeterminedData[$key] = number_format((($item["both_lab_flu_b_nosub"] / $lciPlusData[$key]) * 100), 2, '.', '');
            }

            // calcul of % for each site en function of sum total of the 6 variable
            foreach ($lci_plus as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $totalDataBySite[$key] = $AH1N1pdm09Data[$key] + $AH3Data[$key] + $BYamagataLineageData[$key] + $BVictoriaLineageData[$key] + $ANotSubtypedData[$key] + $BLineageNotDeterminedData[$key];
                $AH1N1pdm09Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($AH1N1pdm09Data[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $AH3Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($AH3Data[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $BYamagataLineageData[$key] = $totalDataBySite[$key] > 0 ? intval(round(($BYamagataLineageData[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $BVictoriaLineageData[$key] = $totalDataBySite[$key] > 0 ? intval(round(($BVictoriaLineageData[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $H1NotherLineageData[$key] = $totalDataBySite[$key] > 0 ? intval(round(($H1NotherLineageData[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $H3NotherLineageData[$key] = $totalDataBySite[$key] > 0 ? intval(round(($H3NotherLineageData[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $ANotSubtypedData[$key] = $totalDataBySite[$key] > 0 ? intval(round(($ANotSubtypedData[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $BLineageNotDeterminedData[$key] = $totalDataBySite[$key] > 0 ? intval(round(($BLineageNotDeterminedData[$key] / $totalDataBySite[$key]) * 100)) : 0;

                $totalPercentPerSite[$key] = $AH1N1pdm09Data[$key] + $AH3Data[$key] + $BYamagataLineageData[$key] + $BVictoriaLineageData[$key] + $H1NotherLineageData[$key] + $H3NotherLineageData[$key] + $ANotSubtypedData[$key] + $BLineageNotDeterminedData[$key];

                //init the biggest value to the first value
                // then we are going to verify all the rest of varibles to get the biggest value
                $biggest = $AH1N1pdm09Data[$key];
                if ($AH3Data[$key] > $biggest) {
                    $biggest = $AH3Data[$key];
                }
                if ($BYamagataLineageData[$key] > $biggest) {
                    $biggest = $BYamagataLineageData[$key];
                }
                if ($BVictoriaLineageData[$key] > $biggest) {
                    $biggest = $BVictoriaLineageData[$key];
                }
                if ($H1NotherLineageData[$key] > $biggest) {
                    $biggest = $H1NotherLineageData[$key];
                }
                if ($H3NotherLineageData[$key] > $biggest) {
                    $biggest = $H3NotherLineageData[$key];
                }
                if ($ANotSubtypedData[$key] > $biggest) {
                    $biggest = $ANotSubtypedData[$key];
                }
                if ($BLineageNotDeterminedData[$key] > $biggest) {
                    $biggest = $BLineageNotDeterminedData[$key];
                }
                // here we are going to check if the total of percent per site
                // if equal to 99 we will add +1 to the biggest value in order to get 100% in the total of percent per site
                if ($totalPercentPerSite[$key] == 99) {
                    switch ($biggest) {
                        case $AH1N1pdm09Data[$key]:
                            $AH1N1pdm09Data[$key] += 1;
                            break;
                        case $AH3Data[$key]:
                            $AH3Data[$key] += 1;
                            break;
                        case $BYamagataLineageData[$key]:
                            $BYamagataLineageData[$key] += 1;
                            break;
                        case $BVictoriaLineageData[$key]:
                            $BVictoriaLineageData[$key] += 1;
                            break;
                        case $H1NotherLineageData[$key]:
                            $H1NotherLineageData[$key] += 1;
                            break;
                        case $H3NotherLineageData[$key]:
                            $H3NotherLineageData[$key] += 1;
                            break;
                        case $ANotSubtypedData[$key]:
                            $ANotSubtypedData[$key] += 1;
                            break;
                        case $BLineageNotDeterminedData[$key]:
                            $BLineageNotDeterminedData[$key] += 1;
                            break;
                    }
                }
                // here we are going to check if the total of percent per site
                // if equal to 100 we will substr -1 to the biggest value in order to get 100% in the total of percent per site
                if ($totalPercentPerSite[$key] == 101) {
                    switch ($biggest) {
                        case $AH1N1pdm09Data[$key]:
                            $AH1N1pdm09Data[$key] -= 1;
                            break;
                        case $AH3Data[$key]:
                            $AH3Data[$key] -= 1;
                            break;
                        case $BYamagataLineageData[$key]:
                            $BYamagataLineageData[$key] -= 1;
                            break;
                        case $BVictoriaLineageData[$key]:
                            $BVictoriaLineageData[$key] -= 1;
                            break;
                        case $H1NotherLineageData[$key]:
                            $H1NotherLineageData[$key] -= 1;
                            break;
                        case $H3NotherLineageData[$key]:
                            $H3NotherLineageData[$key] -= 1;
                            break;
                        case $ANotSubtypedData[$key]:
                            $ANotSubtypedData[$key] -= 1;
                            break;
                        case $BLineageNotDeterminedData[$key]:
                            $BLineageNotDeterminedData[$key] -= 1;
                            break;
                    }
                }
            }
            // change some site names
            $search = ['Ivory Coast', 'Russian Federation - Moscow', 'Russian Federation - Saint Petersburg', 'Turkey', 'Senegal - Dakar'];
            $replace = ['Côte d’Ivoire', 'Russia - Moscow', 'Russia - St Petersburg', 'Türkiye', 'Senegal'];
            $listSite = str_replace($search, $replace, $sites);

            return array(
                'sitesChart' => $listSite,
                'sites' => $sites,
                'lci_plus' => $lciPlusData,
                'totalLciPplus' => $totalLciPplus,
                'AH1N1pdm09' => $AH1N1pdm09Data,
                'AH3' => $AH3Data,
                'BYamagataLineage' => $BYamagataLineageData,
                'BVictoriaLineage' => $BVictoriaLineageData,
                'H1NotherLineage' => $H1NotherLineageData,
                'H3NotherLineage' => $H3NotherLineageData,
                'ANotSubtyped' => $ANotSubtypedData,
                'BLineageNotDetermined' => $BLineageNotDeterminedData,
                'totalPercentPerSite'=>$totalPercentPerSite
            );
        }
    }

    private function querySQLLCI($variable, $season, $sites)
    {
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $querySQL = 'SELECT SUM(' . $variable . ') AS ' . $variable . ', zone FROM (SELECT d.' . $variable . ' AS ' . $variable . ', s.zone, s.id
                            FROM data d 
                            LEFT JOIN site s ON d.site_id = s.id
                            WHERE d.' . $variable . ' = 1 AND s.zone IN ("' . implode("\",\"", $sites) . '")
                            AND d.season IN ("' . implode("\",\"", $season) . '")) AS tmp GROUP BY zone;';
        $resultQuery = $connDB->fetchAll($querySQL);

        return $resultQuery;
    }
    // End  distribution LCI

    // start  distribution ICU
    // ICU : Distribution Intensive care unit
    /**
     * @Route("/init-data-distribution-icu", name="init_data_distribution_icu_ajax")
     */
    public function initDataDistributionICU()
    {
        // get the current season and date start of season
        $result = $this->getCurrentSeason();
        $season = $result['season'];
        // get list of sites valid in the selected season ( at least they have row in table data)
        $listSitesHasDataInSelectedSeason = $this->getlistValidSitesBySeason(explode(',', $season));

        // get data Distribution of lci by season
        $dataDistributionICU = $this->getDataDistributionICU(explode(',', $season), $listSitesHasDataInSelectedSeason);

        return new JsonResponse([
            'status' => 200,
            'current_season' => $season,
            //DATA FOR THE CURVE Distribution LCI
            'lci_plus' => $dataDistributionICU['lci_plus'],
            'both_icu_adm' => $dataDistributionICU['both_icu_adm'],
            'both_death_hosp' => $dataDistributionICU['both_death_hosp'],
            'sites' => $dataDistributionICU['sites'],
            // END DATA CURVE
        ]);
    }

    /**
     * Description: get data of distribution of lci when the user make a filter
     * @Route("/statistics-distribution-icu", name="statistics_distribution_icu_ajax")
     */
    public function dataStatisticsDistributionICU(Request $request)
    {
        // GET DATA FROM THE FORMA DATA IN AJAX
        $data = $request->get('data');
        //LIST OF SEASONS SELECTED
        $seasons = $data['seasons'];
        // LIST Of SITES SELECTED
        $sites = $data['sites'];

        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataDistributionICU(explode(',', $seasons), explode(',', $sites))]);
    }

    // Global function we call to return the data of distribution ICU
    // we give it the list of seasons and the list of sites as parameters
    public function getDataDistributionICU($season, $sites)
    {
        // intialize the result in array format and all data by site to 0
        $lciPlusData = $ICUAdmissionsData = $deathsData = [0];

        if ($season) {
            $lci_plus = $this->querySQLICU('lci_plus', $season, $sites);
            $ICUAdmissions = $this->querySQLICU('both_icu_adm', $season, $sites);
            $deaths = $this->querySQLICU('both_death_hosp', $season, $sites);

            // data lci_plus
            $totalLciPlus = 0;
            foreach ($lci_plus as $item) {
                $totalLciPlus += (int)$item["lci_plus"];
            }
            $lciPlusData[0] = $totalLciPlus;

            // data both_icu_adm
            $totalICUAdmissions = 0;
            foreach ($ICUAdmissions as $item) {
                $totalICUAdmissions += (int)$item["both_icu_adm"];
            }
            $ICUAdmissionsData[0] = $totalICUAdmissions;

            // data both_death_hosp
            $totaldeaths = 0;
            foreach ($deaths as $item) {
                $totaldeaths += (int)$item["both_death_hosp"];
            }
            $deathsData[0] = $totaldeaths;
            return array(
                'sites' => $sites,
                'lci_plus' => $lciPlusData,
                'both_icu_adm' => $ICUAdmissionsData,
                'both_death_hosp' => $deathsData,
            );
        }
    }

    private function querySQLICU($variable, $season, $sites)
    {
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $querySQL = 'SELECT SUM(' . $variable . ') AS ' . $variable . ', zone FROM (SELECT (d.' . $variable . ') AS ' . $variable . ', s.zone, s.id
                            FROM data d 
                            LEFT JOIN site s ON d.site_id = s.id
                            WHERE d.' . $variable . ' = 1  AND s.zone IN ("' . implode("\",\"", $sites) . '")
                            AND d.season IN ("' . implode("\",\"", $season) . '")) AS tmp GROUP BY zone;';
        $resultQuery = $connDB->fetchAll($querySQL);

        return $resultQuery;
    }

    // End  distribution ICU


    // start  distribution EPA
    // EPA : Distribution of enrolled patients by age group - By sites
    /**
     * @Route("/init-data-distribution-epa", name="init_data_distribution_epa_ajax")
     */
    public function initDataDistributionEPA()
    {
        // get the current season and date start of season
        $result = $this->getCurrentSeason();
        $season = $result['season'];
        // get list of sites valid in the selected season ( at least they have row in table data)
        $listSitesHasDataInSelectedSeason = $this->getlistValidSitesBySeason(explode(',', $season));

        // get data Distribution of EPA by season
        $dataDistributionEPA = $this->getDataDistributionEPA(explode(',', $season), $listSitesHasDataInSelectedSeason);

        return new JsonResponse([
            'status' => 200,
            'current_season' => $season,
            //DATA FOR THE CURVE Distribution EPA
            'included' => $dataDistributionEPA['included'],
            'totalIncluded' => $dataDistributionEPA['totalIncluded'],
            'ageLessThan5' => $dataDistributionEPA['ageLessThan5'],
            'ageBetween5And49' => $dataDistributionEPA['ageBetween5And49'],
            'ageBetween50And64' => $dataDistributionEPA['ageBetween50And64'],
            'ageMoreThan65' => $dataDistributionEPA['ageMoreThan65'],
            'sites' => $dataDistributionEPA['sites'],
            'sitesChart' => $dataDistributionEPA['sitesChart'],
            'totalPercentPerSite' => $dataDistributionEPA['totalPercentPerSite'],
            // END DATA CURVE
        ]);
    }

    /**
     * Description: get data of distribution of EPA when the user make a filter
     * @Route("/statistics-distribution-epa", name="statistics_distribution_epa_ajax")
     */
    public function dataStatisticsDistributionEPA(Request $request)
    {
        // GET DATA FROM THE FORMA DATA IN AJAX
        $data = $request->get('data');
        //LIST OF SEASONS SELECTED
        $seasons = $data['seasons'];
        // LIST Of SITES SELECTED
        $sites = $data['sites'];

        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataDistributionEPA(explode(',', $seasons), explode(',', $sites))]);
    }

    // Global function we call to return the data of distribution of EPA
    // we give it the list of seasons and the list of sites as parameters
    public function getDataDistributionEPA($season, $sites)
    {
        /* connexion à la base de données */
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        //  LIST OF SITES BY KEYS
        $sitesKeys = $this->listKeysSites($sites);

        // intialize the result in array format and all data by site to 0
        $totalDataBySite = $includedData = $ageLessThan5Data = $ageBetween5And49Data = $ageBetween50And64Data = $ageMoreThan65Data = $totalPercentPerSite = array_combine(array_keys($sitesKeys), array_fill(0, count($sitesKeys), 0));

        if ($season) {
            $ageLessThan5 = $this->querySQLEPA('< 5y', $season, $sites);
            $ageBetween5And49 = $this->querySQLEPA('5-49y', $season, $sites);
            $ageBetween50And64 = $this->querySQLEPA('50-64y', $season, $sites);
            $ageMoreThan65 = $this->querySQLEPA('+65y', $season, $sites);

            // data included
            $sqlIncluded = 'SELECT SUM(included) AS included, zone FROM (SELECT (d.included) AS included, s.zone, s.id
                            FROM data d 
                            LEFT JOIN site s ON d.site_id = s.id
                            WHERE d.included = 1 AND  s.zone IN ("' . implode("\",\"", $sites) . '") 
                            AND d.season IN ("' . implode("\",\"", $season) . '")) AS tmp GROUP BY zone;';
            $included = $connDB->fetchAll($sqlIncluded);
            $totalIncluded = 0;
            foreach ($included as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $includedData[$key] = (int)$item["included"];
                $totalIncluded += (int)$item["included"];
            }
            // data AH1N1pdm09
            foreach ($ageLessThan5 as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $ageLessThan5Data[$key] = $item["tranche_age_def3"];
            }

            // data AH3
            foreach ($ageBetween5And49 as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $ageBetween5And49Data[$key] = $item["tranche_age_def3"];
            }

            // data  B(Yamagata lineage)
            foreach ($ageBetween50And64 as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $ageBetween50And64Data[$key] = $item["tranche_age_def3"];
            }

            // data  B (Victoria lineage)
            foreach ($ageMoreThan65 as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $ageMoreThan65Data[$key] = $item["tranche_age_def3"];
            }

            foreach ($included as $item) {
                $zone = $item["zone"];
                $key = array_search($zone, $sitesKeys);
                $totalDataBySite[$key] = $ageLessThan5Data[$key] + $ageBetween5And49Data[$key] + $ageBetween50And64Data[$key] + $ageMoreThan65Data[$key];
                $ageLessThan5Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($ageLessThan5Data[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $ageBetween5And49Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($ageBetween5And49Data[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $ageBetween50And64Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($ageBetween50And64Data[$key] / $totalDataBySite[$key]) * 100)) : 0;
                $ageMoreThan65Data[$key] = $totalDataBySite[$key] > 0 ? intval(round(($ageMoreThan65Data[$key] / $totalDataBySite[$key]) * 100)) : 0;

                $totalPercentPerSite[$key] = $ageLessThan5Data[$key] + $ageBetween5And49Data[$key] + $ageBetween50And64Data[$key] + $ageMoreThan65Data[$key];

                // init the biggest value to the first value then we're going to check the rest of variables.
                $biggest = $ageLessThan5Data[$key];
                if ($ageBetween5And49Data[$key] > $biggest) {
                    $biggest = $ageBetween5And49Data[$key];
                }
                if ($ageBetween50And64Data[$key] > $biggest) {
                    $biggest = $ageBetween50And64Data[$key];
                }
                if ($ageMoreThan65Data[$key] > $biggest) {
                    $biggest = $ageMoreThan65Data[$key];
                }
                // here we are going to check if the total of percent per site
                // if equal to 99 we will add +1 to the biggest value in order to get 100% in the total of percent per site
                if ($totalPercentPerSite[$key] == 99) {
                    switch ($biggest) {
                        case $ageLessThan5Data[$key]:
                            $ageLessThan5Data[$key] += 1;
                            break;
                        case $ageBetween5And49Data[$key]:
                            $ageBetween5And49Data[$key] += 1;
                            break;
                        case $ageBetween50And64Data[$key]:
                            $ageBetween50And64Data[$key] += 1;
                            break;
                        case $ageMoreThan65Data[$key]:
                            $ageMoreThan65Data[$key] += 1;
                            break;
                    }
                }
                // here we are going to check if the total of percent per site
                // if equal to 101 we will substr -1 to the biggest value in order to get 100% in the total of percent per site
                if ($totalPercentPerSite[$key] == 101) {
                    switch ($biggest) {
                        case $ageLessThan5Data[$key]:
                            $ageLessThan5Data[$key] -= 1;
                            break;
                        case $ageBetween5And49Data[$key]:
                            $ageBetween5And49Data[$key] -= 1;
                            break;
                        case $ageBetween50And64Data[$key]:
                            $ageBetween50And64Data[$key] -= 1;
                            break;
                        case $ageMoreThan65Data[$key]:
                            $ageMoreThan65Data[$key] -= 1;
                            break;
                    }
                }
            }
            // change some site names
            $search = ['Ivory Coast', 'Russian Federation - Moscow', 'Russian Federation - Saint Petersburg', 'Turkey', 'Senegal - Dakar'];
            $replace = ['Côte d’Ivoire', 'Russia - Moscow', 'Russia - St Petersburg', 'Türkiye', 'Senegal'];
            $listSite = str_replace($search, $replace, $sites);

            return array(
                'sitesChart' => $listSite,
                'sites' => $sites,
                'totalIncluded' => $totalIncluded,
                'included' => $includedData,
                'ageLessThan5' => $ageLessThan5Data,
                'ageBetween5And49' => $ageBetween5And49Data,
                'ageBetween50And64' => $ageBetween50And64Data,
                'ageMoreThan65' => $ageMoreThan65Data,
                'totalPercentPerSite' => $totalPercentPerSite,
            );
        }
    }

    private function querySQLEPA($like, $season, $sites)
    {
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        $querySQL = 'SELECT COUNT(tranche_age_def3) AS tranche_age_def3, zone FROM (SELECT (d.tranche_age_def3) AS tranche_age_def3, s.zone, s.id
                            FROM data d 
                            LEFT JOIN site s ON d.site_id = s.id
                            WHERE  d.included = 1 AND d.tranche_age_def3 LIKE "' . $like . '" AND s.zone IN ("' . implode("\",\"", $sites) . '")
                            AND d.season IN ("' . implode("\",\"", $season) . '")) AS tmp GROUP BY zone;';
        $resultQuery = $connDB->fetchAll($querySQL);
        return $resultQuery;
    }
    // End  distribution EPA


    // start  distribution EPAAS
    // EPAAS : Distribution of enrolled patients by age group – All sites
    /**
     * @Route("/init-data-distribution-epaas", name="init_data_distribution_epaas_ajax")
     */
    public function initDataDistributionEPAAS()
    {
        // get the current season and date start of season
        $result = $this->getCurrentSeason();
        $season = $result['season'];
        // get list of sites valid in the selected season ( at least they have row in table data)
        $listSitesHasDataInSelectedSeason = $this->getlistValidSitesBySeason(explode(',', $season));

        // get data Distribution of EPAAS by season
        $dataDistributionEPAAS = $this->getDataDistributionEPAAS(explode(',', $season), $listSitesHasDataInSelectedSeason);

        return new JsonResponse([
            'status' => 200,
            'current_season' => $season,
            //DATA FOR THE CURVE Distribution LCI
            'totalIncluded' => $dataDistributionEPAAS['totalIncluded'],
            'ageLessThan5' => $dataDistributionEPAAS['ageLessThan5'],
            'ageBetween5And49' => $dataDistributionEPAAS['ageBetween5And49'],
            'ageBetween50And64' => $dataDistributionEPAAS['ageBetween50And64'],
            'ageMoreThan65' => $dataDistributionEPAAS['ageMoreThan65'],
            'sites' => $dataDistributionEPAAS['sites'],
            // END DATA CURVE
        ]);
    }

    /**
     * Description: get data of distribution of epaas when the user make a filter
     * @Route("/statistics-distribution-epaas", name="statistics_distribution_epaas_ajax")
     */
    public function dataStatisticsDistributionEPAAS(Request $request)
    {
        // GET DATA FROM THE FORMA DATA IN AJAX
        $data = $request->get('data');
        //LIST OF SEASONS SELECTED
        $seasons = $data['seasons'];
        // LIST Of SITES SELECTED
        $sites = $data['sites'];

        return new JsonResponse([
            'status' => 200,
            'data' => $this->getDataDistributionEPAAS(explode(',', $seasons), explode(',', $sites))]);
    }

    // Global function we call to return the data of distribution epaas
    // we give it the list of seasons and the list of sites as parameters
    public function getDataDistributionEPAAS($season, $sites)
    {
        /* connexion à la base de données */
        $connDB = \Doctrine\DBAL\DriverManager::getConnection(['url' => $this->getParameter("DATABASE_URL")]);

        // intialize the result in array format and all data to 0
        $ageLessThan5Data = $ageBetween5And49Data = $ageBetween50And64Data = $ageMoreThan65Data = [0];

        if ($season) {

            // data included
            $sqlIncluded = 'SELECT SUM(included) AS included, zone FROM (SELECT (d.included) AS included, s.zone, s.id
                            FROM data d 
                            LEFT JOIN site s ON d.site_id = s.id
                            WHERE d.included = 1 AND  s.zone IN ("' . implode("\",\"", $sites) . '") 
                            AND d.season IN ("' . implode("\",\"", $season) . '")) AS tmp GROUP BY zone;';
            $included = $connDB->fetchAll($sqlIncluded);
            $totalIncluded = 0;
            foreach ($included as $item) {
                $totalIncluded += (int)$item["included"];
            }

            $ageLessThan5 = $this->querySQLEPA('< 5y', $season, $sites);
            $ageBetween5And49 = $this->querySQLEPA('5-49y', $season, $sites);
            $ageBetween50And64 = $this->querySQLEPA('50-64y', $season, $sites);
            $ageMoreThan65 = $this->querySQLEPA('+65y', $season, $sites);

            // data  <5y
            $totalAgeLessThan5 = 0;
            foreach ($ageLessThan5 as $item) {
                $totalAgeLessThan5 += (int)$item["tranche_age_def3"];
            }
            $ageLessThan5Data[0] = $totalAgeLessThan5;

            // data 5-49y
            $totalAgeBetween5And49 = 0;
            foreach ($ageBetween5And49 as $item) {
                $totalAgeBetween5And49 += (int)$item["tranche_age_def3"];
            }
            $ageBetween5And49Data[0] = $totalAgeBetween5And49;

            // data 50-64y
            $totalAgeBetween50And64 = 0;
            foreach ($ageBetween50And64 as $item) {
                $totalAgeBetween50And64 += (int)$item["tranche_age_def3"];
            }
            $ageBetween50And64Data[0] = $totalAgeBetween50And64;

            // data +65y
            $totalAgeMoreThan65 = 0;
            foreach ($ageMoreThan65 as $item) {
                $totalAgeMoreThan65 += (int)$item["tranche_age_def3"];
            }
            $ageMoreThan65Data[0] = $totalAgeMoreThan65;

            return array(
                'sites' => $sites,
                'ageLessThan5' => $ageLessThan5Data,
                'ageBetween5And49' => $ageBetween5And49Data,
                'ageBetween50And64' => $ageBetween50And64Data,
                'ageMoreThan65' => $ageMoreThan65Data,
                'totalIncluded' => $totalIncluded,
            );
        }
    }
    // End  distribution EPAAS

}
